/*
 * (C) Copyright Keith Visco 1998, 1999  All rights reserved.
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * The Copyright owner will not be liable for any damages suffered by
 * you as a result of using the Program. In no event will the Copyright
 * owner be liable for any special, indirect or consequential damages or
 * lost profits even if the Copyright owner has been advised of the
 * possibility of their occurrence.
 */
package com.kvisco.net;


import java.io.InputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.Reader;
import java.net.URL;
import java.net.MalformedURLException;

/**
 * A utility class for URI handling
 * @author <a href="mailto:kvisco@ziplink.net">Keith Visco</a>
**/
public class URIUtils {
    
    /**
     * the File protocol
    **/
    private static final String FILE_PROTOCOL_PREFIX = "file:///";
    
    /**
     * the path separator for an URI
    **/
    private static final char HREF_PATH_SEP = '/';
    
    /**
     * The Device separator for an URI
    **/
    private static final char DEVICE_SEP = '|';
    
	/**
	 * Returns an InputStream for the file represented by the href
	 * argument
	 * @param href the href of the file to get the input stream for.
	 * @param documentBase the document base of the href argument, if it
	 * is a relative href
	 * set documentBase to null if there is none.
	 * @return an InputStream to the desired resource
	 * @exception java.io.FileNotFoundException when the file could not be
	 * found
	**/
	public static InputStream getInputStream(String href, String documentBase) 
	    throws java.io.FileNotFoundException, java.io.IOException
	{
	    
	    //-- check for absolute url
	    URL url = null;
	    try {
	        url = new URL(href);
	        return url.openStream(); 
	    }
	    catch (MalformedURLException muex) {};
	    
	    //-- join document base + href
	    String xHref = null;
	    if ((documentBase != null) && (documentBase.length() > 0)) {
	        int idx = documentBase.lastIndexOf(HREF_PATH_SEP);
	        if (idx == (documentBase.length()-1))
	            xHref = documentBase+href;
	        else
	            xHref = documentBase+HREF_PATH_SEP+href;
	    }   
	    else xHref = href;
	        
	    //-- check for relative url
	    try {
	        url = new URL(xHref);   
	        return url.openStream();
	    }
	    catch(MalformedURLException muex) {};
	    
	    // Try local files
	    File iFile = new File(href);
	    if (iFile.isAbsolute()) return new FileInputStream(iFile);
	    else iFile = new File(xHref);
	    
	    return new FileInputStream(iFile);
	    
	} //-- getInputStream
	
	/**
	 * Returns the document base of the href argument
	 * @return the document base of the given href
	**/
	public static String getDocumentBase(String href) {
	    
	    String docBase = "";
	    
	    if (href == null) return docBase;
	    
	    int idx = -1;
	    //-- check for URL
	    try {
	        URL url = new URL(href);
	        idx = href.lastIndexOf(HREF_PATH_SEP);
	    }
	    catch(MalformedURLException muex) {
	        //-- The following contains a fix from Shane Hathaway 
	        //-- to handle the case when both "\" and "/" appear in filename
            int idx2 = href.lastIndexOf(HREF_PATH_SEP);
	        idx = href.lastIndexOf(File.separator);
            if (idx2 > idx) idx = idx2;	        
	    }
	   
	    if (idx >= 0) docBase = href.substring(0,idx);
	    
	    return docBase;
	} //-- getDocumentBase
	
	/**
	 * 
	**/
	public static String resolveHref(String href, String documentBase) {
	    
	    try {
	        URL url = new URL(href);
	        return href;
	    }
	    catch(MalformedURLException muex) {};

	    
	    //-- join document base + href
	    String xHref = null;
	    if ((documentBase != null) && (documentBase.length() > 0)) {
	        int idx = documentBase.lastIndexOf(HREF_PATH_SEP);
	        if (idx == (documentBase.length()-1))
	            xHref = documentBase+href;
	        else
	            xHref = documentBase+HREF_PATH_SEP+href;
	    }   
	    else xHref = href;
	        
	    //-- check for relative url
	    try {
	        URL url = new URL(xHref);   
	        return xHref;
	    }
	    catch(MalformedURLException muex) {};
	    
	    // Try local files
	    File iFile = new File(href);
	    if (iFile.isAbsolute())
	        xHref = createFileURL(iFile.getAbsolutePath());
	    else {
	        iFile = new File(xHref);
	        xHref = createFileURL(iFile.getAbsolutePath());
	    }
	    return xHref;
	} //-- resolveHref
	
	private static String createFileURL(String filename) {
	    
	    if (filename == null) return FILE_PROTOCOL_PREFIX;
	    int size = filename.length() + FILE_PROTOCOL_PREFIX.length();
	    StringBuffer sb = new StringBuffer(size);
	    sb.append(FILE_PROTOCOL_PREFIX);
	    char[] chars = filename.toCharArray();
	    for (int i = 0; i < chars.length; i++) {
	        char ch = chars[i];
	        switch (ch) {
	            case '\\':
	                sb.append(HREF_PATH_SEP);
	                break;
	            default:
	                sb.append(ch);
	                break;
	                
	        }
	    }
	    return sb.toString();
	} //-- createFileURL
    
} //-- URIUtils