/* HTGET (C) J Whitham 1998
** based on snarf, the Simple Non-interactive All-purpose Resource Fetcher
** snarf is copyright (C) 1995, 1996 Zachary Beane
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is dsitributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILIY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
** General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**
** The author of this program may be reached via email at
** jwhitham@globalnet.co.uk
*/

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <netdb.h>
#include <stdio.h>
#include <time.h>

#define MAXLEN 		256
#define BIG_BUFFER_SIZE 4096
#define BASE64_END 	'='

const char base64_table[64]= "ABCDEFGHIJKLMNOPQRSTUVWXYZ" 
                             "abcdefghijklmnopqrstuvwxyz"
                             "0123456789+/";
			     
#define HEADER_INDICATOR	"TTP"
#define	USER_PASS		"anonymous:altrn@"
#define MY_NAME			"alternate 1"
#define CONTENT_LENGTH		"Content-Length:"

#define MAX_URLS		16

#define OPTIONS			15

/* The HTGet function can return these errors: */
#define ERROR_TIMEOUT		1
#define ERROR_WRITE		2
#define ERROR_ABORT		3
#define ERROR_RESUME_IMPOSSIBLE	4
#define ERROR_HTTP		5
#define ERROR_NOT_NEEDED	6	/* file is already complete */
#define ERROR_TCP		7
#define DOWNLOAD_OK		8
#define ERROR_PARSE		9
#define ERROR_LIST		10	

#define BANNER	"HTGET (c) J Whitham 1998  <jwhitham@globalnet.co.uk> version 0.1\n" \
		"See README. This program comes with NO WARRANTY of any kind.\n\n" 
	
const char	LongOption [ OPTIONS ][ 16 ] = { "--noresume" ,
		"--notimeout" , "--timeout" , "--redial" ,
		"--silent" , "--nocheck" , "--list" , "--logfile" ,
		"--directory" , "--noestimate" , "--minrate" ,
		"--stdout" , "--verbose" , "--be" , "--help" } ;
const char	ShortOption [ OPTIONS ][ 4 ] = { "-nr" ,
		"-nt" , "-t" , "-r" ,
		"-s" , "-nc" , "-l" , "-f" ,
		"-d" , "-ne" , "-m" ,
		"-S" , "-v" , "-b" , "-h" } ;
char	NoResume = 0 ;
long	Timeout = 300 ;		/* 5 minutes default timeout */
char	Redial = 0 ;
char	Silent = 0 ;
char	NoCheck = 0 ;
char	Help = 0 ;
char	ListFile [ MAXLEN ] = { "" } ;
char	LogFile [ MAXLEN ] = { "" } ;
char	Directory [ MAXLEN ] = { "./" } ;
char	NoEstimate = 0 ;
long	MinRate = 0 ;		/* No minimum rate */
char	Verbose = 0 ;
char	StdOut = 0 ;
char	Identity [ MAXLEN ] = { "linux htget v0.1alpha" } ;
FILE	* ReportsOutTo ;


char * MakeHTTPRequest ( char * Hostname , char * Filename ) ;
int TCPConnect ( char * Hostname , unsigned int Port ) ;
int ReadLine ( int Socket , char * ReceiveBuffer ) ;
char * ToBase64(unsigned char *bin, int len ) ;
int ProcessURL ( char * TheURL , char * Hostname , char * Filename , char * ActualFilename , unsigned * Port ) ;
int GetHeaderForRequest ( unsigned socket , char * request , char * Header ) ;
void PrintReport ( long ArrivedSize , long RangeSize , 
	time_t CurrentTime , time_t StartTime , time_t TimeoutTime ) ;
			
			

int main ( int argc , char * argv [] )
{
	int		I , Port , J , Success , Ok , AllowResume , Again ;
	char		RetrieveURL [ MAX_URLS ][ MAXLEN ] ;
	char		Parameter [ MAXLEN ] ;
	int		URLs , CurrentURL ;
	FILE		* ListFileHandle ;

	URLs = 0 ;
	for ( I = 1 ; I < argc ; I ++ )
	{
		if ( argv [ I ][ 0 ] == '-' )
		{
			for ( J = 0 ; J < OPTIONS ; J ++ )
			{
				if ( ( I + 1 ) < argc )
				{
					strcpy ( Parameter , argv [ I + 1 ] ) ;
				} else {
					strcpy ( Parameter , "" ) ;
				}
				if (( strcmp ( LongOption [ J ] , argv [ I ] ) == 0 )
				|| ( strcmp ( ShortOption [ J ] , argv [ I ] ) == 0 ))
				{
					switch ( J )
					{
						case 0 :	NoResume = 1 ;
								break ;
						case 1 :	Timeout = 0 ;
								break ;
						case 2 :	Timeout = atol ( Parameter ) ;
								I ++ ;
								break ;
						case 3 :	Redial = 1 ;
								break ;
						case 4 :	Silent = 1 ;
								break ;
						case 5 :	NoCheck = 1 ;
								break ;
						case 6 :	strcpy ( ListFile , Parameter ) ;
								I ++ ;
								break ;
						case 7 :	strcpy ( LogFile , Parameter ) ;
								I ++ ;
								break ;
						case 8 :	strcpy ( Directory , Parameter ) ;
								I ++ ;
								break ;
						case 9 :	NoEstimate = 1 ;
								break ;
						case 10 :	MinRate = atol ( Parameter ) ;
								I ++ ;
								break ;
						case 11 :	StdOut = 1 ;
								break ;
						case 12 :	Verbose = 1 ;
								break ;
						case 13 :	strcpy ( Identity , Parameter ) ;
								I ++ ;
								break ;
						case 14 :	Help = 1 ;
								break ;
					}
					break ;
				}
			}
		} else {
			if ( strlen ( argv [ I ] ) > 0 )
			{
				strcpy ( RetrieveURL [ URLs ] , argv [ I ] ) ;
				URLs ++ ; 
				if ( URLs >= MAX_URLS )
				{
					fprintf ( stderr , "The maximum number of URLs that can be specified at the command line is %d.\n" , MAX_URLS ) ;
					return ( 0 ) ;
				}
			}
		}
	}
	if ( (( URLs < 1 ) && ( strlen ( ListFile ) < 1 ) ) || Help )  
	{
		fprintf ( stderr , "Usage: htget <urls..> [options]\n\n"
			"--help		-h	This text\n"
			"--noresume	-nr	Do not attempt to resume files.\n"
			"--notimeout	-nt	Never let the connection time out.\n"
			"--timeout <t>	-t <t>	Set the timeout to <t> seconds.\n"
			"--redial	-r	If the connection times out, reconnect and resume\n"
			"			the download instead of returning to the console.\n" 
			"--silent	-s	Dont output download info to stderr.\n"
			"--nocheck	-nc	When resuming files, dont check the file time/dates\n"
			"			match.\n"
			"--list <l>	-l <l>	Download files listed in file <l>.\n"
			"--logfile <l>	-f <l>	Make a log file <l>\n"
			"--directory <d> -d <d>	Output files to alternate directory <d>\n"
			"--noestimate	-ne	Dont estimate the download time remaining\n"
			"--minrate <r>	-m <r>	Set the minimum download rate to <r> bytes/sec.\n"
			"--stdout	-S	Output file to stdout. Only works with one file.\n"
			"--verbose	-v	Lots of information including HTTP headers\n"
			"--be <n>	-b <n>	Set the user-agent to <n>\n\n" ) ;
		return ( 0 ) ;
	}
	fprintf ( stderr , BANNER ) ;

	if ( StdOut && ( URLs > 1 ))
	{
		fprintf ( stderr , "You can't send multiple files to stdout.\n" ) ;
		return ( 0 ) ;
	}
	if ( strlen ( LogFile ) > 1 )
	{
		ReportsOutTo = fopen ( LogFile , "at" ) ;
		if ( ReportsOutTo == NULL )
		{
			fprintf ( stderr , "Couldn't open log file %s\n" , LogFile ) ;
			return ( 0 ) ;	
		}
		fprintf ( ReportsOutTo , "\n" ) ;
		fprintf ( ReportsOutTo , BANNER ) ;
		Silent = 0 ;		/* because there is no
			output to the screen anyway */ 
		Verbose = 1 ;	
	} else {
		ReportsOutTo = stderr ;
	}
	Ok = 1 ;
	AllowResume = 1 ;
	CurrentURL = 0 ;
	Success = 0 ;
	while (( Ok ) && ( CurrentURL < URLs ))
	{
		if ( ! Silent )
		{
			fprintf ( ReportsOutTo , "\n" ) ;
		}
		Ok = 1 ;
		switch ( HTGet ( RetrieveURL [ CurrentURL ] , AllowResume ) )
		{
			case ERROR_TIMEOUT :	/* just try again */
						AllowResume = 1 ;
						break ;
			case ERROR_WRITE :	if ( ! Silent )
						{
							fprintf ( ReportsOutTo , "\nError writing to local file!\n" ) ;
						}	
						CurrentURL ++ ;
						break ;
			case ERROR_RESUME_IMPOSSIBLE :	/* I hate this! */
						AllowResume = 0 ;
						break ;
			case ERROR_NOT_NEEDED :	
						CurrentURL ++ ;
						break ;
			case ERROR_TCP :	/* All of these are unrecoverable errors */
			case ERROR_ABORT :	Ok = 0 ;
						break ;
			case ERROR_HTTP :	CurrentURL ++ ;
						break ;
			case ERROR_PARSE :	CurrentURL ++ ;	/* next please */
						break ;
			case DOWNLOAD_OK :	Success ++ ;
						CurrentURL ++ ;	
						break ;
		}				
	}
	if ( strlen ( ListFile ) > 0 )
	{
		ListFileHandle = fopen ( ListFile , "rt" ) ;
		if ( ListFileHandle == NULL )
		{
			if ( ! Silent )
			{
				fprintf ( ReportsOutTo , "Couldn't open list file %s\n" , ListFile ) ;
			} 
		} else {	 
			Ok = 1 ;
			Again = 0 ;
			while ( Ok )
			{
				if ( ! Silent )
				{
					fprintf ( ReportsOutTo , "\n" ) ;
				}
				if ( ! Again )
				{
					if ( feof ( ListFileHandle ) )
					{
						Ok = 0 ;
						I = ERROR_LIST ;
					} else {
						fgets ( RetrieveURL [ 0 ] , MAXLEN - 2 , ListFileHandle ) ;
						for ( I = strlen ( RetrieveURL [ 0 ] ) - 1 ; I >= 0 ; I -- )
						{
							if ( ( isalpha ( RetrieveURL [ 0 ][ I ] ))
							|| ( isdigit ( RetrieveURL [ 0 ][ I ] )) )
							{
								break ;
							} else {	
								RetrieveURL [ 0 ][ I ] = '\0' ;
							}
						}
						if ( strlen ( RetrieveURL [ 0 ] ) > 0 )
						{
							I = HTGet ( RetrieveURL [ 0 ] , AllowResume ) ;
						} else {
							I = ERROR_LIST ;
						}
					}	
				} else {
					I = HTGet ( RetrieveURL [ 0 ] , AllowResume ) ;
				}
				Again = 1 ;
				switch ( I )
				{
					case ERROR_TIMEOUT :	/* just try again */
								AllowResume = 1 ;
								break ;
					case ERROR_WRITE :	if ( ! Silent )
								{
									fprintf ( ReportsOutTo , "\nError writing to local file!\n" ) ;
								}	
								Again = 0 ;
								break ;
					case ERROR_RESUME_IMPOSSIBLE :	/* I hate this! */
								AllowResume = 0 ;
								break ;
					case ERROR_NOT_NEEDED :	
								Again = 0 ;
								break ;
					case ERROR_TCP :	/* All of these are unrecoverable errors */
					case ERROR_ABORT :	Ok = 0 ;
								break ;
					case ERROR_HTTP :	Again = 0 ;
								break ;
					case ERROR_PARSE :	Again = 0 ;
								break ;
					case DOWNLOAD_OK :	Success ++ ;
								Again = 0 ;
								break ;
					case ERROR_LIST :	Ok = 0 ;
								break ;
				}				
			}
			fclose ( ListFileHandle ) ;
		}
	}
			
	if ( ! Silent )
	{
		fprintf ( ReportsOutTo , "\n\nSuccessfully downloaded %d files\n" ,
			Success ) ;
	}		
	if ( ReportsOutTo != stderr )
	{
		fclose ( ReportsOutTo ) ;
	}
	return ( 1 ) ;
}

int HTGet ( char * RetrieveURL , int AllowResume ) 
{
	/* request-header */
	char		Header [ BIG_BUFFER_SIZE ] ;
	char		Request [ BIG_BUFFER_SIZE ] ;
	/* tcp/ip */
	unsigned	socket ;
	/* URL */
	int		I , Port ;
	char		ch ;
	char		Hostname [ MAXLEN ] ;
	char		Filename [ MAXLEN ] ;
	char		ActualFilename [ MAXLEN ] ;
	char		Parameter [ MAXLEN ] ;
	int		Resuming = 0 ;
	/* output */
	FILE		* File ;
	long		ActualSize , ArrivedSize ;
 	char 		recv_buf [ BIG_BUFFER_SIZE ] ;
	int		Percentage ;
	long		CurrentSize ;
	long		RangeSize = 0 , RangeStart = 0 ;
	/* authenticate */
	char		* UserPass ;
	struct stat	ConnectionStatistics ;
	/* timing */
	time_t		StartTime , CurrentTime , TimeoutTime ;
	time_t		RedundantReportsTimer ;


	UserPass = (char *) malloc ( strlen ( USER_PASS ) + 1 ) ;
	strcpy ( UserPass , USER_PASS ) ;
	
	if ( ! ProcessURL ( RetrieveURL , Hostname , Filename , ActualFilename , & Port ) )
	{
		if ( ! Silent )
		{
			fprintf ( ReportsOutTo , "Couldn't parse URL %s\n" , RetrieveURL ) ;
		}	
		return ( ERROR_PARSE ) ;
	}

	/* apply alternate directory stuff */
	for ( I = strlen ( Directory ) - 1 ; I >= 0 ; I -- )
	{
		if ( Directory [ I ] == '/' )
		{
			Directory [ I ] == '\0' ;
		} else {
			break ;
		}
	}
	strcat ( Directory , "/" ) ;
	strcpy ( recv_buf , ActualFilename ) ;
	strcpy ( ActualFilename , Directory ) ;
	strcat ( ActualFilename , recv_buf ) ;
	
	CurrentSize = 0 ;
	if ( ! StdOut )
	{
		/* read/resume check */
		File = fopen ( ActualFilename , "rb" ) ;
		if ( File != NULL )
		{
			fseek ( File , 0 , SEEK_END ) ;
			CurrentSize = ftell ( File ) ;
			fclose ( File ) ;
		}
		/* write check */
		File = fopen ( ActualFilename , "ab" ) ;
		if ( File == NULL )
		{
			if ( ! Silent )
			{
				fprintf ( ReportsOutTo , "Error: attempt to open file %s for writing failed.\n" , ActualFilename ) ;
			}
			return ( ERROR_WRITE ) ;
		}
		fclose ( File ) ;
	}
	ActualSize = -1 ;
	/* NoResume is a command line switch.
	AllowResume is set false if the resume fails 1st time */
	if (( CurrentSize > 0 ) && ( ! NoResume ) && ( ! StdOut ) && ( AllowResume ))
	{
		/* See if resume is possible by finding out the file length,
		time and date */
		socket = TCPConnect ( Hostname , Port ) ;
		if ( socket == 0 )
		{
			return ( ERROR_TCP ) ; /* error messages already generated by TCPConnect */
		}
	
		fprintf ( ReportsOutTo , "Getting header for %s.." , RetrieveURL ) ;
		sprintf ( Request , "HEAD %s HTTP/1.0\r\n"
			"Host: %s\r\n"
			"User-agent: %s\r\n"
			"Authorization: Basic %s\r\n\r\n" ,
			    Filename , Hostname , Identity ,
			    ToBase64 ( UserPass , strlen ( UserPass ) ) ) ;
		if ( Verbose )
		{
			fprintf ( ReportsOutTo , "\n** Sent: %s" , Request ) ;
		}
		if ( ! GetHeaderForRequest ( socket , Request , Header ) )
		{
			return ( ERROR_HTTP ) ;
		}
		if ( Verbose )
		{
			fprintf ( ReportsOutTo , "\n** Received: %s" , Header ) ;
		}
		/* get some data from the header..
		 * the actual size for file resumes */
		Resuming = 0 ;
		if ( GetHeaderData ( Header , "Content-Length:" , recv_buf ) )
		{
			ActualSize = atol ( recv_buf ) ;
			if ( CurrentSize > 0 )
			{
				if ( CurrentSize < ActualSize )
				{
					Resuming = 1 ;
					if ( ! Silent )
					{
						fprintf ( ReportsOutTo , "Resuming download from %ld bytes\n" , CurrentSize ) ;
					}	
				}
				if ( CurrentSize == ActualSize )
				{
					if ( ! Silent )
					{
						fprintf ( ReportsOutTo , "Remote and local size is the same!\n" ) ;
					}	
					return ( ERROR_NOT_NEEDED ) ;
				}
			}
		}
		/* next bit: is it really necessary to reopen the socket?? */
	} else {
		CurrentSize = 0 ;
		/* Avoid bugs in the next bit */
	}
		

	socket = TCPConnect ( Hostname , Port ) ;
	if ( socket == 0 )
	{
		return ( ERROR_TCP ) ;
	}
	if ( ! Silent )
	{
		fprintf ( ReportsOutTo , "Getting data for %s.." , RetrieveURL ) ;
	}	
	if ( Resuming )
	{
		sprintf ( Request , "GET %s HTTP/1.1\r\n"
			"Host: %s\r\n"
			"User-agent: %s\r\n" 
			"Range: bytes=%ld-%ld\r\n"
			"Authorization: Basic %s\r\n\r\n" ,
				Filename , Hostname , Identity ,
				CurrentSize , ActualSize ,
				ToBase64 ( UserPass , strlen ( UserPass ) ) ) ;
	} else {
		sprintf ( Request , "GET %s HTTP/1.0\r\n"
			"Host: %s\r\n"
			"User-agent: %s\r\n" 
			"Authorization: Basic %s\r\n\r\n" ,
				Filename , Hostname , Identity ,
				ToBase64 ( UserPass , strlen ( UserPass ) ) ) ;
	}
	if ( Verbose )
	{
		fprintf ( ReportsOutTo , "\n** Sent: %s" , Request ) ;
	}
	if ( ! GetHeaderForRequest ( socket , Request , Header ) )
	{
		return ( ERROR_HTTP ) ;
	}

	if ( Verbose )
	{
		fprintf ( ReportsOutTo , "\n** Received: %s" , Header ) ;
	}
	if ( Resuming )
	{
		RangeSize = ActualSize - CurrentSize ;
	} else {
		RangeSize = ActualSize ;
	}
	if ( GetHeaderData ( Header , "Content-Length:" , recv_buf ) )
	{
		RangeSize = atol ( recv_buf ) ;
		if ( ActualSize < 0 )
		{
			ActualSize = RangeSize ;
		}
	}
	RangeStart = 0 ;
	if ( GetHeaderData ( Header , "Content-Range:" , recv_buf ) )
	{
		/* example from RFC:
		 * Content-Range: bytes 21010-47021/47022
		 * add 6 bytes */
		RangeStart = atol ( & recv_buf [ 6 ] ) ;
	}
	if (( Resuming )
	&& (( RangeStart > CurrentSize ) || ( RangeStart < 128 )) )
	{
		fprintf ( ReportsOutTo , "This server does not support resume\n" ) ;
		return ( ERROR_RESUME_IMPOSSIBLE ) ;
	}

	if ( ! StdOut )
	{
		if ( CurrentSize > 0 )
		{
			File = fopen ( ActualFilename , "ab" ) ;
		} else {
			File = fopen ( ActualFilename , "wb" ) ;
		}
		if ( File == NULL )
		{
			return ( ERROR_WRITE ) ;
		}
	}
	if ( RangeStart > CurrentSize )
	{
		fprintf ( ReportsOutTo , "The HTTP range is past the end of file\n" ) ;
		return ( ERROR_RESUME_IMPOSSIBLE ) ;
			/* well actually it isn't but this will force the
			gizmo to try again from 0 bytes */
	}
	if ( ! StdOut )
	{
		fseek ( File , RangeStart , SEEK_SET ) ;
	}
 
 	
	fprintf ( ReportsOutTo , "\nReceiving HTTP data.." ) ;
	I = read ( socket , recv_buf , sizeof ( recv_buf ) ) ;
	fcntl ( socket , F_SETFL , O_NONBLOCK ) ;
	
	ArrivedSize = 0 ;
	fprintf ( ReportsOutTo , "..\nTotal size: %ld bytes\n" , ActualSize ) ;
	StartTime = time ( NULL ) ;
	TimeoutTime = StartTime + Timeout ;
	while ( I > 0 )
	{	
		if ( StdOut )
		{
			fwrite ( recv_buf , I , 1 , stdout ) ;
		} else {
			if ( fwrite ( recv_buf , I , 1 , File ) != 1 )
			{
				fprintf ( ReportsOutTo , "\nError writing to file %s\n" , ActualFilename ) ;
				fclose ( File ) ;
				return ( ERROR_WRITE ) ;
			}
		}
		ArrivedSize += I ;
		CurrentTime = time ( NULL ) ;
		if ( MinRate < 1 )
		{
			TimeoutTime = CurrentTime + Timeout ;
		} else {
			/* BytesPerSecond = ArrivedSize / ( CurrentTime - StartTime ) ; */
			if (( ArrivedSize > 0 ) 
			&& ( CurrentTime > StartTime ))
			{
				if (( ArrivedSize / ( CurrentTime - StartTime )) < MinRate )
				{
					/* just not good enough */
				} else {
					TimeoutTime = CurrentTime + Timeout ;
				}
			} else {
				TimeoutTime = CurrentTime + Timeout ;
			}
		}
		I = -1 ;
		RedundantReportsTimer = 0 ;
		while ( I < 0 )
		{
			I = read ( socket , recv_buf , sizeof ( recv_buf ) ) ;
			if (( ! Silent )
			&& ( ReportsOutTo == stderr )
			&& ( RedundantReportsTimer != CurrentTime ))
			{
				PrintReport ( ArrivedSize , RangeSize , CurrentTime , StartTime , TimeoutTime ) ;
				RedundantReportsTimer = CurrentTime ;
			}
			CurrentTime = time ( NULL ) ;
			if ( Timeout < 1 )
			{
				TimeoutTime = CurrentTime + 120 ;
			}
			if ( CurrentTime > TimeoutTime )
			{
				if ( ! StdOut )
				{
					fclose ( File ) ;
				}
				return ( ERROR_TIMEOUT ) ;
			}
		}
	}
	if ( ! StdOut )
	{
		fclose ( File ) ;
	}
	if ( ! Silent )
	{
		fprintf ( ReportsOutTo , "\nFinished downloading.\n" ) ;
	}
	return ( DOWNLOAD_OK ) ;
}


/* MakeHTTPRequest
 * Makes a HTTP Request header for Hostname/Filename 
 * [ZB] */
char * MakeHTTPRequest ( char * Hostname , char * Filename )
{
	char	tmp [ MAXLEN ] ;
	char	*request ;
	char	*pwstring ;

	/* size of the two strings, plus ':' and trailing null */

	pwstring = (char *) malloc ( strlen ( USER_PASS ) + 1 ) ;

	strcpy ( pwstring , USER_PASS ) ;

	sprintf(tmp, "GET %s HTTP/1.0\r\n"
		"Host: %s\r\n"
		"User-agent: snarf %s\r\n"
		"Authorization: Basic %s\r\n\r\n",
		Filename , Hostname , MY_NAME ,
		ToBase64 ( pwstring , strlen ( pwstring ) ) ) ;
	request = (char *) malloc ( sizeof ( tmp ) + 1 ) ;
	strncpy ( request , tmp , MAXLEN ) ;
	return ( request ) ;
}

/* TCPConnect
 * Gets a socket number for the host after finding it's address
 * [ZB] */
int TCPConnect ( char * Hostname , unsigned int Port )
{
	struct hostent		* host ;
	struct sockaddr_in	sin ;
	int			sock_fd ;

	if( ( host = gethostbyname ( Hostname ) ) == NULL ) 
	{
		if ( ! Silent )
		{
			fprintf ( ReportsOutTo , "Error finding host %s\n" , Hostname ) ;
		}
		return ( 0 ) ;
	}

/* get the socket */
	if( ( sock_fd = socket ( AF_INET , SOCK_STREAM , 0 ) ) < 0 )
	{
		if ( ! Silent )
		{
			fprintf ( ReportsOutTo , "Error creating TCP/IP socket\n" ) ;
		}
		return ( 0 ) ;
	}

  /* connect the socket, filling in the important stuff */
	sin . sin_family = AF_INET ;
	sin . sin_port = htons ( Port ) ;
	memcpy ( & sin . sin_addr , host -> h_addr , host -> h_length ) ;
  
	if ( connect ( sock_fd , ( struct sockaddr * ) & sin , sizeof ( sin ) ) < 0 )
	{
		if ( ! Silent )
		{
			fprintf ( ReportsOutTo , "Error connecting to %s\n" , Hostname ) ;
		}
		return ( 0 ) ;
	}

	return ( sock_fd ) ;
}

/* ReadLine
 * Reads a \n terminated line (includes the \n)
 * [ZB] heavily modified by jw.*/
 
int ReadLine ( int Socket , char * ReceiveBuffer )
{
	int	rc = 0 ;
	char	ch ;
	int	I = 0 ;

	rc = read ( Socket , & ch , 1 ) ;
	while ( rc == 1 )
	{
		ReceiveBuffer [ I ] = ch ;
		I ++ ;
		if ( ch == '\n' )
		{
			break ;
		}
		rc = read ( Socket , & ch , 1 ) ;
	}
	if ( rc == 0 )
	{
		if ( I == 0 )
		{
			/* nothing read.. */
			return ( 0 ) ;
		}
		/* unexpected EOF */
	}
	if (( rc != 0 ) && ( rc != 1 ))
	{
		/* something has gone very wrong!? */
		return ( -1 ) ;
	}
	ReceiveBuffer [ I ] = '\0' ;
	return ( I ) ;
}
			
	
/* ToBase64 [ZB]
 * lauri alanko, i thank you */
char * ToBase64(unsigned char *bin, int len)
{
	char *buf= (char *)malloc((len+2)/3*4+1);
	int i=0, j=0;
	
	while(j<len-2){
		buf[i++]=base64_table[bin[j]>>2];
		buf[i++]=base64_table[((bin[j]&3)<<4)|(bin[j+1]>>4)];
		buf[i++]=base64_table[((bin[j+1]&15)<<2)|(bin[j+2]>>6)];
		buf[i++]=base64_table[bin[j+2]&63];
		j+=3;
	}
	switch(len-j){
	    case 1:
		buf[i++]=base64_table[bin[j]>>2];
		buf[i++]=base64_table[(bin[j]&3)<<4];
		buf[i++]=BASE64_END;
		buf[i++]=BASE64_END;
		break;
	    case 2:
		buf[i++]=base64_table[bin[j]>>2];
		buf[i++]=base64_table[((bin[j]&3)<<4)|(bin[j+1]>>4)];
		buf[i++]=base64_table[(bin[j+1]&15)<<2];
		buf[i++]=BASE64_END;
		break;
	    case 0:
	}
	buf[i]='\0';
	return buf;
}

/* ProcessURL
 * Converts TheURL to it's component parts. Uses defaults if necessary.
 * [JW] returns 0 if error. 1 if OK. */
int ProcessURL ( char * TheURL , char * Hostname , char * Filename , char * ActualFilename , unsigned * Port )
{
	char		BufferURL [ MAXLEN ] ;
	char		NormalURL [ MAXLEN ] ;
	int		I ;
	
	/* clear everything. Is this necessary? not sure */
	memset ( Hostname , '\0' , strlen ( TheURL ) ) ;
	memset ( Filename , '\0' , strlen ( TheURL ) ) ;
	memset ( ActualFilename , '\0' , strlen ( TheURL ) ) ;
	/* remove "http://" */
	strcpy ( BufferURL , TheURL ) ;
	/* BUGFIX: strlwr ( BufferURL )? */
	if ( strncmp ( BufferURL , "http://" , 7 ) == 0 )
	{
		strcpy ( NormalURL , & TheURL [ 7 ] ) ;
	} else {
		strcpy ( NormalURL , TheURL ) ;
	}
	if ( strlen ( NormalURL ) < 1 )
	{
		return ( 0 ) ;
	}
	/* scan for first /. */
	for ( I = 0 ; I < strlen ( NormalURL ) ; I ++ )
	{
		if ( NormalURL [ I ] == '/' )
		{
			break ;
		}
	}
	/* whatever, I will be the end of the hostname */
	strncpy ( Hostname , NormalURL , I ) ;
	if ( I < 1 )
	{
		return ( 0 ) ; /* gone wrong -- too short */
	}
	strcpy ( BufferURL , & NormalURL [ I ] ) ;
	if ( strlen ( BufferURL ) < 1 )
	{
		/* no filename error! */
		return ( 0 ) ;
	}
	(* Port) = 80 ; /* default HTTP port */
	for ( I = strlen ( BufferURL ) - 1 ; I >= 0 ; I -- )
	{
		if ( BufferURL [ I ] == ':' )
		{
			/* everything after I is a port number */
			(* Port) = (unsigned int) atol ( & BufferURL [ I + 1 ] ) ;
			BufferURL [ I ] = '\0' ;
			break ;
		}
	}
	/* that means that BufferURL is the Filename */
	strcpy ( Filename , BufferURL ) ;
	if ( strlen ( BufferURL ) < 1 )
	{
		/* no filename error, again */
		return ( 0 ) ;
	}
	/* now the ActualFilename, everything after the last / */
	for ( I = strlen ( Filename ) - 1 ; I >= 0 ; I -- )
	{
		if ( Filename [ I ] == '/' )
		{
			strcpy ( ActualFilename , & Filename [ I + 1 ] ) ;
			break ;
		}
	}
	if ( strlen ( ActualFilename ) < 1 )
	{
		return ( 0 ) ; 
		/* because that means this is a directory
		download. And I can't do them. */
	}
	return ( 1 ) ;
}

/* GetHeaderForRequest .. gets the HTTP header for
 * request @ socket into Header */
int GetHeaderForRequest ( unsigned socket , char * request , char * Header )
{
	int		Timeout , I ;
	char		line [ MAXLEN ] ;
	char		status [ MAXLEN ] ;
	char		token [ MAXLEN ] ;
	int 		in_header;
	char		ch ;

	Header [ 0 ] = '\0' ;
	send ( socket , request , strlen ( request ) , 0 ) ;

	/* hmm...let's do a bit of status checking by reading the numeric
	response code from the server */

	Timeout = 0 ;

	/* I don't know how much of this waiting stuff is really needed,
	but I put it in after some problems connecting to my local server,
	there are possibly some latency issues here? */
	
	if ( ! Silent )
	{
		fprintf ( ReportsOutTo , ".." ) ;
	}
	ch = '\0' ;
	I = read ( socket , & ch , 1 ) ;
	while ( I == 0 )
	{
		if ( ! Silent )
		{
			fprintf ( ReportsOutTo , "." ) ;
		}
		sleep ( 1 ) ;	
		I = read ( socket , & ch , 1 ) ;
	}

	if ( I == -1 )
	{
		if ( ! Silent )
		{
			fprintf ( ReportsOutTo , "\nUnknown TCP/IP error\n" ) ;
		}
		return ( 0 ) ;
	}

	strcpy ( status , "" ) ;
	if ( ! ReadLine ( socket , status ) )
	{
		if ( ! Silent )
		{
			fprintf ( ReportsOutTo , "no response\n" )  ;
		}
		return ( 0 ) ;
	} else {
		if ( ! Silent )
		{
			fprintf ( ReportsOutTo , "\n" ) ;
		}
	}
	/* this will re-append the check character read above to the 1st line */
	line [ 0 ] = ch ;
	line [ 1 ] = '\0' ;
	strcat ( line , status ) ;
	strcpy ( status , line ) ;

	/* scan line for "HTTP" to indicate this is the header */
	for ( I = 0 ; I < ( strlen ( line ) - strlen ( HEADER_INDICATOR ) ) ; I ++ )
	{
		if ( strncmp ( & line [ I ] , HEADER_INDICATOR , strlen ( HEADER_INDICATOR ) ) == 0 )
		{
			in_header = 1 ;
			break ;
		}
	}
	
	if ( in_header )
	{
		strtok(line, " ");
		strcpy(token, (char *)strtok(NULL, " "));
		if ( token [ 0 ] == '4' || token [ 0 ] == '5' )
		{
			if ( ! Silent )
			{
				fprintf ( ReportsOutTo , "http error from server: %s" , status ) ;
			}
			return(0);
		}
	}
		
	while ( in_header )
	{
		if ( ! ReadLine(socket, line) ) 
		{
			in_header = 0;
			break;
		}

		if ( line [ 0 ] == '\n' || line [ 1 ] == '\n' )
		{
			in_header = 0;

		}
		strcat ( Header , line ) ;
	}
	return ( 1 ) ;
}


int NoCaseStrNCmp ( char * String1 , char * String2 , int Length )
{
	int	I ;
	
	for ( I = 0 ; I < Length ; I ++ )
	{
		if ( toupper ( String1 [ I ] ) != toupper ( String2 [ I ] ))
		{
			return ( 0 ) ;	
		}
	}
	return ( 1 ) ;
}

int GetHeaderData ( char * Header , char * FieldName , char * Record )
{
	int	I , J ;

	for ( I = 0 ; I < strlen ( Header ) ; I ++ )
	{
		if ( NoCaseStrNCmp ( & Header [ I ] , FieldName , strlen ( FieldName ) ) )
		{
			I += strlen ( FieldName ) ;
			strcpy ( Record , & Header [ I ] ) ;
			for ( J = 0 ; J < strlen ( Record ) ; J ++ )
			{
				if ( Record [ J ] == '\n' )
				{
					Record [ J ] = '\0' ;
					break ;
				}
			}
			return ( 1 ) ;
		}
	}
	return ( 0 ) ;
}


void PrintReport ( long ArrivedSize , long RangeSize , 
	time_t CurrentTime , time_t StartTime , time_t TimeoutTime )
{
	long		Percentage ;
	long		BytesPerSecond ;
	time_t		TimeLeft ;
	long		BytesLeft ;

	if ( RangeSize > 0 )
	{
		Percentage = ( ArrivedSize * 100 ) / RangeSize ;
		fprintf ( stderr , "\r\033[K\r%ld/%ld (%d%%)" , ArrivedSize , RangeSize , Percentage ) ;
	} else {
		fprintf ( stderr , "\r\033[K\r%ld/unknown" , ArrivedSize ) ;
	}
	if ( ( CurrentTime - StartTime ) > 0 )
	{
		BytesPerSecond = ArrivedSize / ( CurrentTime - StartTime ) ;
		fprintf ( stderr , " %ld bytes/sec" , BytesPerSecond ) ;
		if (( RangeSize > 0 ) && ! NoEstimate )
		{
			/* time left may be predicted. */
			BytesLeft = RangeSize - ArrivedSize ;
			TimeLeft = BytesLeft / BytesPerSecond ;
			if ( TimeLeft > 3599 )
			{
				/* hours and minutes */
				fprintf ( stderr , "%4dh %02dm" ,
					( TimeLeft / 3600 ) , (( TimeLeft / 60 ) % 60 ) ) ;
			} else {
				fprintf ( stderr , "%4dm %02ds" ,
					(( TimeLeft / 60 ) % 60 ) ,
					( TimeLeft % 60 ) ) ;
			}
		}
	}
	if (( CurrentTime < TimeoutTime )
	&& ( CurrentTime > ( TimeoutTime - 30 )))
	{
		fprintf ( stderr , " Timeout in %lds" , ( TimeoutTime - CurrentTime ) ) ;
	}
}
