/*
 *      Copyright (C) 1997 Claus-Justus Heine

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *
 * $Source: /homes/cvs/ftape-stacked/contrib/ftformat/ftfmt-tapelib.c,v $
 * $Revision: 1.19.6.3 $
 * $Date: 1997/11/23 21:07:28 $
 *
 *      This program contains the tape manipulation functions for the
 *      user level floppy tape formatting stuff for the
 *      QIC-40/80/3010/3020 floppy-tape driver "ftape" for Linux.
 */

#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#ifndef MAP_FAILED
#define MAP_FAILED (void *)(-1)
#endif
#include <stdio.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include <time.h>

#include <linux/ftape.h>
#include <linux/ftape-vendors.h>
#include <linux/ftape-header-segment.h>

#include "ftformat.h"
#include "ftfmt-options.h"
#include "ftfmt-bsm.h"
#include "ftfmt-tapelib.h"

const ftape_error                 qic117_error[] = QIC117_ERRORS;
const struct qic117_command_table qic117_cmd[]   = QIC117_COMMANDS;

void *tape_mmap(int tape_fd, size_t dma_size)
{
	void *dma_buffer;
	struct mtop  mtop;

	/*
	 *  First try to set the number of buffers to the desired size:
	 */
	mtop.mt_op    = MTSETDRVBUFFER;
	mtop.mt_count = dma_size / FT_BUFF_SIZE;
	if (ioctl(tape_fd, MTIOCTOP, &mtop) == -1) {
		perror("Error setting tape drive buffering");
		return NULL;
	}
	/*
	 *  Then try to mmap it
	 */
	dma_buffer = mmap(0, dma_size, PROT_READ|PROT_WRITE, MAP_SHARED,tape_fd,0);
	if (dma_buffer == MAP_FAILED) {
		perror("Error mapping tape dma buffers");
		(void)close(tape_fd);
		return NULL;
	}
#if 0
	printf("Got mmaped tape buffer @ %p/%d\n", dma_buffer, dma_size);
#endif
	return dma_buffer;
}

/*
 *  open the tape device, try to determine whether it is a floppy tape,
 *  and store the hardware status in drive_config, resp. tape_config.
 *  mmap() the dma buffers if op_mode != PROBING
 *  
 *  Returns the tape fd on success, -1 otherwise
 *
 *  Note: we first open the tape device read-only, as the tape type
 *  autodetection of some drives doesn't work with damaged cartridges.
 *  Later, after we have told the tape drive which format mode we
 *  need, we re-open it read-write
 *
 */
int tape_reopen_rw(const char *name, int tape_fd)
{
	if (tape_fd == -1) {
		fprintf(stderr, "%s hasn't been opened yet!\n", name);
		return -1;
	}
	if (close(tape_fd) == -1) {
		perror("Error closing tape dev");
		return -1; /* is it still open, or not ? */
	}
	if ((tape_fd = open(name, O_RDWR)) == -1) {
		perror("Error re-opening tape dev read/write");
		return -1;
	}
	return tape_fd;
}

int tape_open(const char *name, struct opt_parms *opts,
			  u_int8_t *drive_status,
			  u_int8_t *drive_config,
			  u_int8_t *tape_status)
{
	int tape_fd;
	struct mtget mtget;
	struct mtop  mtop = { MTNOP, 1 };
	ft_drive_status hw_status;
	static vendor_struct vendors[] = QIC117_VENDORS;
	vendor_struct *vendor;

	/* open the tape device
	 */
	if ((tape_fd = open(name, O_RDONLY)) == -1) {
		perror("Error opening tape device read-only");
		return -1;
	}
	if (opts->read_hseg) {
		/* set its status, might result in reading the header segments.
		 */
		if (ioctl(tape_fd, MTIOCTOP, &mtop) == -1) {
			perror("Error setting tape drive status");
			(void)close(tape_fd);
			return -1;
		}
	}
	/* get its status
	 */
	if (ioctl(tape_fd, MTIOCGET, &mtget) == -1) {
		perror("Error getting tape drive status");
		(void)close(tape_fd);
		return -1;
	}
	if (GMT_DR_OPEN(mtget.mt_gstat)) {
		fprintf(stderr, "Error: No tape cartridge present!\n");
		(void)close(tape_fd);
		return -1;
	}
	if (GMT_WR_PROT(mtget.mt_gstat)) {
		fprintf(stderr, "Error: Write protected cartridge!\n");
		(void)close(tape_fd);
		return -1;
	}
	if (!GMT_ONLINE(mtget.mt_gstat)) {
		fprintf(stderr, "Error: Tape drive is offline!\n");
		(void)close(tape_fd);
		return -1;
	}
	if ((mtget.mt_type & MT_ISFTAPE_FLAG) == 0) {
		fprintf(stderr, "Error: This is not a floppy tape drive!\n");
		(void)close(tape_fd);
		return -1;
	}
	mtget.mt_type &= ~MT_ISFTAPE_FLAG; /* clear the flag bit */
	if (opts->verbose > 0) {
		vendor = &vendors[0];
		while (vendor->vendor_id != UNKNOWN_VENDOR &&
			   vendor->vendor_id != mtget.mt_type) {
			vendor++;
		}
		printf("Tape drive type: %s (0x%04lx)\n", vendor->name, mtget.mt_type);
	}
	hw_status.space = mtget.mt_dsreg;
	*drive_status   = hw_status.status.drive_status;
	*drive_config   = hw_status.status.drive_config;
	*tape_status    = hw_status.status.tape_status;
	return tape_fd;
}

/*  Close the tape device. I wonder whether we should put it offline,
 *  or reset it straight forward. But why should we? At least this should
 *  ONLY be done after a successful format.
 */
void tape_close(const int tape_fd, void *dma_buffer, const size_t dma_size)
{
	if (tape_fd != -1) {
		/* just in case */
		(void)qic_simple_command(tape_fd,
								 0, QIC_ENTER_PRIMARY_MODE, -1, 0, NULL);
		if (dma_buffer != NULL) {
			if (munmap(dma_buffer, dma_size) == -1) {
				perror("Error unmapping dma buffer");
			}
		}
		if (close(tape_fd) == -1) {
			perror("Error closing tape dev");
		}
	}
}

int qic_simple_command(const int tape_fd,
					   const int ready_wait,
					   const qic117_cmd_t qic_cmd, const int parameter,
					   const int timeout, u_int8_t *status)
{
	struct mtftcmd cmd;

	memset(&cmd, 0, sizeof(cmd));
	cmd.ft_wait_before = ready_wait * 1000;
	cmd.ft_cmd         = qic_cmd;
	if (!UNSET(parameter)) {
		cmd.ft_parm_cnt    = 1;
		cmd.ft_parms[0]    = parameter;
	}
	cmd.ft_wait_after  = timeout * 1000;

	if (ioctl(tape_fd, MTIOCFTCMD, &cmd) == -1) {
		fprintf(stderr, "Ioctl error sending %s command: %s\n",
				qic117_cmd[qic_cmd].name, strerror(errno));
		return -1;
	}
	if (status) {
		*status = cmd.ft_status & 0xff;
	}
	if (cmd.ft_error) {
		fprintf(stderr, "Tape drive error sending %s command: %d (%s)\n",
				qic117_cmd[qic_cmd].name, cmd.ft_error,
				qic117_error[(cmd.ft_error < NR_ITEMS(qic117_error) ?
							  cmd.ft_error : 0)].message);
	}
	return cmd.ft_error ? -1 : 0;
}

int qic_calibrate_tape_length(const int tape_fd)
{
	struct mtftcmd cmd;

	memset(&cmd, 0, sizeof(cmd));
	cmd.ft_cmd         = QIC_CALIBRATE_TAPE_LENGTH;
	cmd.ft_wait_after  = 1300*1000;/*FIXME: calculate timeouts in user space */
	if (ioctl(tape_fd, MTIOCFTCMD, &cmd) == -1) {
		perror("Ioctl error sending QIC_CALIBRATE_TAPE_LENGTH");
		return -1;
	}
	if (cmd.ft_error) {
		fprintf(stderr, "Calibrate tape length not supported. "
				"Surprise! Error: %d (%s)\n",
				cmd.ft_error,
				qic117_error[(cmd.ft_error < NR_ITEMS(qic117_error) ?
							  cmd.ft_error : 0)].message);
		return -1;
	}
	return qic_report_command(tape_fd, QIC_REPORT_FORMAT_SEGMENTS, 16);
}

int qic_report_command(const int tape_fd,
					   const qic117_cmd_t qic_cmd,
					   const int numbits)
{
	struct mtftcmd cmd;

	memset(&cmd, 0, sizeof(cmd));
	cmd.ft_cmd         = qic_cmd;
	cmd.ft_result_bits = numbits;
	if (ioctl(tape_fd, MTIOCFTCMD, &cmd) == -1) {
		fprintf(stderr, "Ioctl error sending %s command: %s\n",
				qic117_cmd[qic_cmd].name, strerror(errno));
		return -1;
	}
	return cmd.ft_result;
}


int write_reference_burst(const int tape_fd)
{
	u_int8_t status;

	printf("Writing reference bursts ... ");
	if (qic_simple_command(tape_fd, 0, QIC_ENTER_FORMAT_MODE, -1, 0, NULL)) {
		fprintf(stderr, "\n"__FUNCTION__ " failed!\n");
		return -1;
	}
	if (qic_simple_command(tape_fd, 0, QIC_WRITE_REFERENCE_BURST, -1, 940,
						   &status)) {
		return -1;
	}
	if (!(status & QIC_STATUS_REFERENCED)) {
		fprintf(stderr, "\n"__FUNCTION__ " failed!\n");
		return -1;
	}
	printf("done.\n");
	return 0;
}

/* Erase the entire cartridge, using physical forward/revers in format mode.
 */
int erase_cartridge(const int tape_fd, const int tpc)
{
	int i;
	u_int8_t status;

	/* the following could be replaced by a MTREW command */
	if (qic_simple_command(tape_fd,
						   0, QIC_PHYSICAL_REVERSE, -1, 650, &status)) {
		return -1;
	}
	if (!(status & QIC_STATUS_AT_BOT)) {
		fprintf(stderr, "Unable to rewind the cartridge\n");
		return -1;
	}
	if (qic_simple_command(tape_fd, 650, QIC_ENTER_FORMAT_MODE, -1, 0, NULL)) {
		return -1;
	}
    printf("Erasing track   0");

	for (i=0; i < tpc; i++) {
		printf("\b\b\b%3d", i);
		if (qic_simple_command(tape_fd,
							   0, QIC_SEEK_HEAD_TO_TRACK, i, 15, &status)) {
			printf("\n");
			return -1;
		}
		if (i&1) {
			if (!(status & QIC_STATUS_AT_EOT)) {
				fprintf(stderr, "\ncartridge not a eot\n");
				return -1;
			}
			if (qic_simple_command(tape_fd,
								   0, QIC_PHYSICAL_REVERSE, -1, 650,&status)) {
				printf("\n");
				return -1;
			}
		} else {
			if (!(status & QIC_STATUS_AT_BOT)) {
				fprintf(stderr, "\ncartridge not a bot\n");
				return -1;
			}
			if (qic_simple_command(tape_fd,
								   0, QIC_PHYSICAL_FORWARD, -1, 650,&status)) {
				printf("\n");
				if (i == 0) {
					/* tape drive doesn't support this command in format mode
					 */
					fprintf(stderr,
							"Please ignore the error message printed above!");
					return 0;
				} else {
					return -1;
				}
			}
		}
	}
	printf("\n");
	return 0;
}

int read_header_segment(const int tape_fd, void *hseg)
{
		struct mtftseg ft_seg;
		int i;
		const unsigned long hseg_magic = FT_HSEG_MAGIC;

		printf("Reading old header segment ... ");
		for (i = 0; i < 64; i++) {
			memset(&ft_seg, 0, sizeof(ft_seg));
			ft_seg.mt_data  = hseg;
			ft_seg.mt_segno = i;
			ft_seg.mt_mode  = MT_FT_RD_SINGLE;
			if (ioctl(tape_fd, MTIOCRDFTSEG, &ft_seg) == -1) {
				fprintf(stderr, "Ioctl error reading header segment: %s\n",
						strerror(errno));
				return -1;
			}
			if (ft_seg.mt_result == FT_SEGMENT_SIZE) {
				break;
			}
		}
		if (memcmp(&hseg_magic, hseg, 4)) {
			fprintf(stderr, "Unable to read the header segment\n");
			fprintf(stderr, "Consider using \"--header-discard\"\n");
			return -1;
		}
		printf("done.\n");
		return 0;
}

/* This function informs the tape drive and the kernel driver about
 * the desired format. It also computes segments_per_floppy_head which
 * is needed for fill_dma_buffer() to compute the data to fed the fdc
 * and tape drive with
 */
int set_format_parms(const int tape_fd, const struct format_parms *fmt_opts)
{
	const struct ftfmtparms *parms = &fmt_opts->fmtparms;
	int fmt;
	struct mtftcmd cmd;
	struct mtftformat fmtc;

	/* first the hardware stuff
	 */
	fmt = (((parms->ft_qicstd & QIC_TAPE_STD_MASK) << 2) |
		   (parms->ft_qicstd & QIC_TAPE_WIDE ? 3 : 1));
	if (qic_simple_command(tape_fd, 0, QIC_SELECT_RATE, fmt, 1, NULL) != 0) {
		fprintf(stderr,
				"Please ignore the error message printed above if it proves\n"
				"that your cartridge can be formatted successfully!\n");
	}
	memset(&cmd, 0, sizeof(cmd));
	cmd.ft_wait_before = 0;
	cmd.ft_cmd         = QIC_SET_FORMAT_SEGMENTS;
	cmd.ft_parm_cnt    = 3;
	cmd.ft_parms[0]    = parms->ft_spt & 0xf;
	cmd.ft_parms[1]    = (parms->ft_spt>>4) & 0xf;
	cmd.ft_parms[2]    = (parms->ft_spt>>8) & 0xf;
	if (ioctl(tape_fd, MTIOCFTCMD, &cmd) == -1) {
		fprintf(stderr, "Ioctl error sending %s command: %s\n",
				qic117_cmd[QIC_SET_FORMAT_SEGMENTS].name, strerror(errno));
		return -1;
	}
	if (cmd.ft_error) {
		fprintf(stderr, "Tape drive error sending %s command: %d (%s)\n",
				qic117_cmd[QIC_SET_FORMAT_SEGMENTS].name, cmd.ft_error,
				qic117_error[(cmd.ft_error < NR_ITEMS(qic117_error) ?
							  cmd.ft_error : 0)].message);
		fprintf(stderr,
				"Please ignore the error message printed above if it proves\n"
				"that your cartridge can be formatted successfully.\n");
	}
	/* and now the software kernel driver stuff
	 */
	memset(&fmtc, 0, sizeof(fmtc));
	fmtc.fmt_op = FTFMT_SET_PARMS;
	fmtc.fmt_arg.fmt_parms = *parms;
	if (ioctl(tape_fd, MTIOCFTFORMAT, &fmtc) == -1) {
		fprintf(stderr, "Ioctl error sending format parameters: %s\n",
				strerror(errno));
		return -1;
	}
	return 0;
}

/* Query the kernel driver about the parms it uses. Needed for --verify-only
 */
int get_format_parms(const int tape_fd, struct format_parms *fmt_opts)
{
	struct ftfmtparms *parms = &fmt_opts->fmtparms;
	struct mtftformat fmtc;

	memset(&fmtc, 0, sizeof(fmtc));
	fmtc.fmt_op = FTFMT_GET_PARMS;
	if (ioctl(tape_fd, MTIOCFTFORMAT, &fmtc) == -1) {
		fprintf(stderr, "Ioctl error getting format parameters: %s\n",
				strerror(errno));
		return -1;
	}
	/*  This is enough. gap3 is only needed for formatting
	 */
	*parms = fmtc.fmt_arg.fmt_parms;
	return 0;
}

/*  This should actually be even easier than in kernel space.
 *  The dma buffer is one large contiguous region.
 */

/*  *user_pos is always (#<last computed segment> + 1). kernel_pos gives
 *  the segment that is currently being formatted by the kernel driver.
 *  We should not touch that segment (i.e. the part of the dma buffer 
 *  that belongs that segment). At the very begin, kernel_pos is 0. We always 
 *  compute segs_in_buf (approx 256) segments in advance, which is plenty.
 *
 *  When starting a new track, we need to start at the beginning of the dma 
 *  buffer
 *
 *  When calling msync() (to support parallel port tape drives with
 *  their funky mmap() implementation) we need to make sure that the
 *  start-address is page-aligned, otherwise msync() will fail.
 *
 */
static void fill_dma_buffer(format_segment *buf, const size_t dma_size,
							int *user_pos, const int kernel_pos,
							const struct ftfmtparms *parms)
{
	int segs_in_buf         = dma_size / sizeof(format_segment);
	int seg_id              = *user_pos;
	int trk_seg             = seg_id % parms->ft_spt;
	format_segment *buf_end = &buf[segs_in_buf];
	format_segment *buf_pos = &buf[trk_seg % segs_in_buf];
	void *buf_start 	= (void*) buf_pos;
	int i;
	static size_t msync_offset = 0;
	size_t msync_size;

	while (seg_id < (kernel_pos + segs_in_buf) && trk_seg < parms->ft_spt) {
		buf_pos->sectors[0].cyl =
			(seg_id % SPFH(parms->ft_ftm)) / SEGMENTS_PER_FLOPPY_TRACK;
		buf_pos->sectors[0].head =
			seg_id / SPFH(parms->ft_ftm);
		buf_pos->sectors[0].sect =
			(seg_id % SEGMENTS_PER_FLOPPY_TRACK) * FT_SECTORS_PER_SEGMENT + 1;
		buf_pos->sectors[0].size = 3;
#if 0
		if (verbose >= 2) {
			fprintf(stderr, "%d %d\n", seg_id, trk_seg);
			fprintf(stderr, "0x%08x\n", *((int *)&buf_pos->sectors[0]));
		}
#endif
		for (i = 1; i < FT_SECTORS_PER_SEGMENT; i++) {
			buf_pos->sectors[i] = buf_pos->sectors[0];
			buf_pos->sectors[i].sect += i;
#if 0
			if (verbose >= 2) {
				fprintf(stderr, "0x%08x\n", *((int *)&buf_pos->sectors[i]));
			}
#endif
		}
		seg_id    ++;
		trk_seg   ++;
		buf_pos   ++; /* pointer arithmetic */

		/* We only sync after computing enough parameters for an entire page,
		 * or when we hit the end of a track.
		 *
		 * Note that end and start of the entire buffer are alway page
		 * aligned as long as PAGE_SIZE < 32k
		 *
		 * Of course, the computation of msync_size is highly inefficient.
		 * We could just treat it as a counter and increment it etc.
		 */
		msync_size = (void*)buf_pos - (void *)buf - msync_offset;
		if (msync_size && (msync_size % getpagesize()) == 0) {
			if (msync((void *)buf + msync_offset, msync_size, MS_SYNC) == -1) {
				fprintf(stderr, "msync(%p, %d, MS_SYNC) failed: %s.\n",
						(void *)buf + msync_offset, msync_size,
						strerror(errno));
				fprintf(stderr, "Dumping core\n");
				fflush(stderr);
				abort();
			}
			msync_offset += msync_size;
		}
		if (buf_pos == buf_end) { /* cycle */
			msync_offset = 0; /* reset to start of buffer */
			buf_start = (void*) buf_pos = buf;
		}
	}
	/* We only sync after computing enough parameters for an entire page,
	 * or when we hit the end of a track.
	 *
	 * Note that end and start of the entire buffer are alway page
	 * aligned as long as PAGE_SIZE < 32k
	 *
	 * The following handles the end-of-tape-track condition:
	 */
	if (trk_seg == parms->ft_spt) {
		msync_size = (void*)buf_pos - (void *)buf - msync_offset;
		if (msync_size) {
			if (msync((void *)buf + msync_offset, msync_size, MS_SYNC) == -1) {
				fprintf(stderr, "msync(%p, %d, MS_SYNC) failed: %s.\n",
						(void *)buf + msync_offset, msync_size,
						strerror(errno));
				fprintf(stderr, "Dumping core\n");
				fflush(stderr);
				abort();
			}
		}
		msync_offset = 0; /* reset to start of buffer */
	}
	*user_pos = seg_id;
}


/*  This does the main work. We don't need to issue raw QIC-117 commands here
 *  because the driver need some information about the track it is located on.
 *  Does it? Maybe change it. First: let's try to make it work again.
 */
int format_cartridge(const int tape_fd, void *dma_buf, const size_t dma_size,
					 const struct format_parms *fmt_opts)
{
	const struct ftfmtparms *parms = &fmt_opts->fmtparms;
	int user_pos = 0, kernel_pos = 0;
	int trk;
	struct mtftformat fmtc;

	if (qic_simple_command(tape_fd, 0, QIC_ENTER_FORMAT_MODE, -1, 0, NULL)) {
		fprintf(stderr, __FUNCTION__ " failed!\n");		
		return -1;
	}
	trk = 0;
	printf("\rFormatting track %3d, ", trk);
	while (trk < parms->ft_tpc) {
		user_pos = kernel_pos = trk * parms->ft_spt;
		fill_dma_buffer(dma_buf, dma_size, &user_pos, kernel_pos, parms);
		printf("segment %8d", kernel_pos);
		memset(&fmtc, 0, sizeof(fmtc));
		fmtc.fmt_op = FTFMT_FORMAT_TRACK;
		fmtc.fmt_arg.fmt_track.ft_track = trk;
		fmtc.fmt_arg.fmt_track.ft_gap3  = fmt_opts->gap3;
		if (ioctl(tape_fd, MTIOCFTFORMAT, &fmtc) == -1) {
			fprintf(stderr, "\nIoctl error formatting track %d: %s\n",
					trk, strerror(errno));
			return -1;
		}
		memset(&fmtc, 0, sizeof(fmtc));
		fmtc.fmt_arg.fmt_status.ft_segment = trk * parms->ft_spt;
		do {
			kernel_pos = fmtc.fmt_arg.fmt_status.ft_segment;
			if (user_pos % parms->ft_spt) {
				fill_dma_buffer(dma_buf, dma_size, &user_pos,kernel_pos,parms);
			}
			if (kernel_pos % parms->ft_spt) {
				/* don't print the first segment of a track 'cause already
				 * done
				 */
				printf("\b\b\b\b\b\b\b\b%8d", kernel_pos);
			}
			fmtc.fmt_op = FTFMT_STATUS;
			if (ioctl(tape_fd, MTIOCFTFORMAT, &fmtc) == -1) {
				fprintf(stderr, "\nError getting format status (%d/%d): %s\n",
						trk, kernel_pos, strerror(errno));
				return -1;
			}
		} while(fmtc.fmt_arg.fmt_status.ft_segment > kernel_pos);
		if (fmtc.fmt_arg.fmt_status.ft_segment - parms->ft_spt * (trk + 1)) {
			printf("\rRetrying   track %3d, ", trk);
		} else {
		    trk ++;
			if (trk < parms->ft_tpc) {
				printf("\rFormatting track %3d, ", trk);
			}
		}
	}
	printf("\n");
	return 0;
}

/*  This does the main work. We don't need to issue raw QIC-117 commands here
 *  because the driver need some information about the track it is located on.
 *  Does it? Maybe change it. First: let's try to make it work again.
 */
int verify_cartridge(const int tape_fd,
					 const struct format_parms *fmt_opts, void *hseg)
{
	const struct ftfmtparms *parms = &fmt_opts->fmtparms;
	int trk, seg;
	struct mtftformat fmtc;
	
	if (qic_simple_command(tape_fd, 0, QIC_ENTER_VERIFY_MODE, -1, 1, NULL)) {
		return -1;
	}
	if (qic_simple_command(tape_fd, 0, QIC_SEEK_LOAD_POINT, -1, 670, NULL)) {
		return -1;
	}	
	for (trk = 0; trk < parms->ft_tpc; trk++) {
		printf("\rVerifying  track %3d, segment %8d", trk, trk*parms->ft_spt);
		for (seg = 0; seg < parms->ft_spt; seg ++) {
			memset(&fmtc, 0, sizeof(fmtc));
			fmtc.fmt_op = FTFMT_VERIFY;
			fmtc.fmt_arg.fmt_verify.ft_segment = trk * parms->ft_spt + seg;
			if (ioctl(tape_fd, MTIOCFTFORMAT, &fmtc) == -1) {
				fprintf(stderr, "\nIoctl error verifying segment %d: %s\n",
						fmtc.fmt_arg.fmt_verify.ft_segment, strerror(errno));
#if 0
				/* the kernel driver now handles totally damaged segments
				 *
				 * but return -EIO if wait_segment() fails.
				 */
				return -1;
#else
				/*  We treat this is an all bad segment.
				 */
				put_bsm_entry(fmtc.fmt_arg.fmt_verify.ft_segment,
							  EMPTY_SEGMENT);
#endif
			} else {
			put_bsm_entry(fmtc.fmt_arg.fmt_verify.ft_segment,
						  fmtc.fmt_arg.fmt_verify.ft_bsm);

			}
#if 0
			if (fmtc.fmt_arg.fmt_verify.ft_bsm != 0) {
				fprintf(stderr, "bsm for segment %d: 0x%08x\n",
					   fmtc.fmt_arg.fmt_verify.ft_segment,
					   fmtc.fmt_arg.fmt_verify.ft_bsm);
			}
#endif
			printf("\b\b\b\b\b\b\b\b%8d", fmtc.fmt_arg.fmt_verify.ft_segment);
		}
	}
	if (qic_simple_command(tape_fd, 0, QIC_ENTER_PRIMARY_MODE, -1, 0, NULL)) {
		printf("\n");
		return -1;
	}
	printf("\n");
	return 0;
}

unsigned long tape_capacity(const struct format_parms *fmt_opts)
{
	int i;
	int seg_size;
	unsigned long capacity;

	capacity = 0;
	for (i = 0;
		 i < fmt_opts->fmtparms.ft_spt * fmt_opts->fmtparms.ft_tpc;
		 i ++) {
		seg_size = (FT_SECTORS_PER_SEGMENT - 
					count_ones(get_bsm_entry(i)) - 
					FT_ECC_SECTORS);
		if (seg_size > 0) {
			capacity += seg_size;
		}
	}
	return capacity;
}

/*  tune the bad sector map, i.e. maps some segments as empty etc.
 *  Returns -EIO if cartridge is really damaged.
 */

/* Segments with more than this many bad sectors are marked as
 * entirely unusable
 */
#define FTFMT_MAX_BAD_SECTORS 8  /*  other suggestions? */

static int tune_bsm(void *hseg,	int *first, int *scnd, int *vtbl, int *last,
					const struct format_parms *fmt_opts)
{
	const struct ftfmtparms *parms = &fmt_opts->fmtparms;
	int i;

	/*  map regions with hole imprints bad for QIC-3010/3020 
	 */
	if ((parms->ft_qicstd & QIC_TAPE_STD_MASK) == QIC_TAPE_QIC3020 ||
	    (parms->ft_qicstd & QIC_TAPE_STD_MASK) == QIC_TAPE_QIC3010) {
		if (parms->ft_qicstd & QIC_TAPE_WIDE) {
			/* 0.315in tape */
			for (i = 17; i <= 37; i += 2) {
				put_bsm_entry(i     * parms->ft_spt,     EMPTY_SEGMENT);
				put_bsm_entry(i     * parms->ft_spt + 1, EMPTY_SEGMENT);
				put_bsm_entry(i     * parms->ft_spt + 2, EMPTY_SEGMENT);
				put_bsm_entry(i     * parms->ft_spt + 3, EMPTY_SEGMENT);
				put_bsm_entry((i+1) * parms->ft_spt - 4, EMPTY_SEGMENT);
				put_bsm_entry((i+1) * parms->ft_spt - 3, EMPTY_SEGMENT);
				put_bsm_entry((i+1) * parms->ft_spt - 2, EMPTY_SEGMENT);
				put_bsm_entry((i+1) * parms->ft_spt - 1, EMPTY_SEGMENT);
			}
		} else {
			/* quarterinch tape */
			for (i = 5; i <= 27; i += 2) {
				put_bsm_entry(i     * parms->ft_spt,     EMPTY_SEGMENT);
				put_bsm_entry(i     * parms->ft_spt + 1, EMPTY_SEGMENT);
				put_bsm_entry(i     * parms->ft_spt + 2, EMPTY_SEGMENT);
				put_bsm_entry(i     * parms->ft_spt + 3, EMPTY_SEGMENT);
				put_bsm_entry((i+1) * parms->ft_spt - 4, EMPTY_SEGMENT);
				put_bsm_entry((i+1) * parms->ft_spt - 3, EMPTY_SEGMENT);
				put_bsm_entry((i+1) * parms->ft_spt - 2, EMPTY_SEGMENT);
				put_bsm_entry((i+1) * parms->ft_spt - 1, EMPTY_SEGMENT);
			}
		}
	}
	/*  Ok. What remains to do: find three segments without errors
	 *  towards the beginning of the tape and write the space between them 
	 *  (if any) with deleted data marks.
	 */
	for (*first = 0; *first < parms->ft_spt * parms->ft_tpc; (*first)++) {
		if (get_bsm_entry(*first) == 0)
			break;
		put_bsm_entry(*first, EMPTY_SEGMENT);
	}
	/* now for the second header segment: */
	for (*scnd= (*first) + 1; *scnd < parms->ft_spt*parms->ft_tpc; (*scnd)++) {
		if (get_bsm_entry(*scnd) == 0)
			break;
		put_bsm_entry(*scnd, EMPTY_SEGMENT);
	}
	/* and now for the volume table segment */
	for (*vtbl = (*scnd) + 1; *vtbl < parms->ft_spt*parms->ft_tpc; (*vtbl)++) {
		if (get_bsm_entry(*vtbl) == 0)
			break;
		put_bsm_entry(*vtbl, EMPTY_SEGMENT);
	}
	if (*vtbl >= parms->ft_spt * parms->ft_tpc) {
		/* totally damaged cartridge */
		fprintf(stderr,
			"Newly formatted cartride hasn't three error free segments.\n");
		return -1;
	}
	/*  Now tune the bad sector map. Travers the map, mark segments with
	 *  more than, say, FTFMT_MAX_BAD_SECTORS as empty
	 *  Also mark sectors where both neighbours are bad as bad.
	 */
	for (i = (*vtbl) + 1; i < parms->ft_spt * parms->ft_tpc; i++) {
		SectorMap map = get_bsm_entry(i);
		if (map != 0) {
			SectorMap newmap = map | ((map >> 1) & (map << 1));
			SectorMap blockmap = (newmap >> 1) & newmap;

			/* surround contigous blocks of bad sectors by one extra bad
			 * sector at each side.
			 */
			newmap  |= (blockmap >> 1) | (blockmap << 2);
			
			if (count_ones(newmap) >= FTFMT_MAX_BAD_SECTORS) {
				newmap = EMPTY_SEGMENT;
			}
			
			if (newmap != map) {
				put_bsm_entry(i, newmap);
			}
		}
	}
	*last = parms->ft_spt * parms->ft_tpc - 1;
	while (*last > *vtbl && get_bsm_entry(*last) == EMPTY_SEGMENT) {
		(*last) --;
	}
	return 0;
}

static u_int32_t create_time_stamp(void)
{
	struct tm *fmt_time;
	time_t raw_time;
	
	time(&raw_time);
	fmt_time = gmtime(&raw_time);
	return FT_TIME_STAMP(fmt_time->tm_year + 1900, 
						 fmt_time->tm_mon, 
						 fmt_time->tm_mday,
						 fmt_time->tm_hour,
						 fmt_time->tm_min,
						 fmt_time->tm_sec);
}


static void compose_header_segment(u_int8_t *hseg, const u_int8_t *tape_label,
								   const struct ftfmtparms *parms,
								   const int first, const int scnd,
								   const int vtbl,  const int last)
{
	u_int32_t time_stamp;

	PUT4(hseg, FT_SIGNATURE, FT_HSEG_MAGIC);
	hseg[FT_FMT_CODE] = parms->ft_fmtcode;
	if (parms->ft_fmtcode == fmt_big) {
		memset(hseg + FT_HSEG_1, 0, 8);
		PUT4(hseg, FT_6_HSEG_1,   first);
		PUT4(hseg, FT_6_HSEG_2,   scnd);
		PUT4(hseg, FT_6_FRST_SEG, vtbl);
		PUT4(hseg, FT_6_LAST_SEG, last);
	} else {
		memset(hseg + FT_6_HSEG_1, 0, 16);
		PUT2(hseg, FT_HSEG_1,   first);
		PUT2(hseg, FT_HSEG_2,   scnd);
		PUT2(hseg, FT_FRST_SEG, vtbl);
		PUT2(hseg, FT_LAST_SEG, last);
	}
	time_stamp = create_time_stamp();
	PUT4(hseg, FT_FMT_DATE, time_stamp);
	PUT4(hseg, FT_WR_DATE,  time_stamp);
	PUT2(hseg, FT_SPT, parms->ft_spt);
	hseg[FT_TPC] = parms->ft_tpc;
	hseg[FT_FHM] = parms->ft_fhm;
	hseg[FT_FTM] = parms->ft_ftm;
	hseg[FT_FSM] = MAX_FLOPPY_SECTOR;
	memset(hseg + FT_LABEL, ' ', FT_LABEL_SZ);
	memcpy(hseg + FT_LABEL, tape_label, strlen(tape_label));
	PUT4(hseg, FT_LABEL_DATE, time_stamp);
	PUT2(hseg, FT_CMAP_START, 0);

	/*  number of segments written, formatted or verified through
	 *  lifetime of tape
	 */
	PUT4(hseg, FT_SEG_CNT,
		 GET4(hseg, FT_SEG_CNT) + (parms->ft_spt * parms->ft_tpc) * 2 + vtbl);
	PUT2(hseg, FT_FMT_CNT, GET2(hseg, FT_FMT_CNT) + 1);
	PUT2(hseg, FT_FSL_CNT, 0); /* clear the failed sector log */
}

static int write_segment(const int tape_fd,
						 const int seg_id, const int mode, void *data)
{
	struct mtftseg ft_seg;
	
	memset(&ft_seg, 0, sizeof(ft_seg));
	ft_seg.mt_data  = data;
	ft_seg.mt_segno = seg_id;
	ft_seg.mt_mode  = mode;
	if (ioctl(tape_fd, MTIOCWRFTSEG, &ft_seg) == -1) {
		fprintf(stderr, "Ioctl error reading header segment: %s\n",
				strerror(errno));
			return -1;
	}
	if (ft_seg.mt_result < 0) {
		fprintf(stderr, "Failed to write segment %d, mode %d: %d\n",
				seg_id, mode, ft_seg.mt_result);
		return -1;
	}
	return 0;
}

int write_header_segments(const int tape_fd,
						  u_int8_t *hseg, const char *tape_label,
						  const struct format_parms *fmt_opts)
{
	const struct ftfmtparms *parms = &fmt_opts->fmtparms;
	int seg_id;
	int first, scnd, vtbl, last;

	if (qic_simple_command(tape_fd, 0, QIC_ENTER_PRIMARY_MODE, -1, 0, NULL)) {
		return -1;
	}
	if (tune_bsm(hseg, &first, &scnd, &vtbl, &last, fmt_opts)) {
		return -1;
	}
	compose_header_segment(hseg, tape_label, parms, first, scnd, vtbl, last);
	/* Ok, now write it out to tape
	 */
	seg_id = 0;
	while (seg_id < first) {
		printf("Deleting segment %d ... ", seg_id);
		if (write_segment(tape_fd, seg_id++, MT_FT_WR_DELETE, NULL))
			return -1;
		printf("done\n");
	}
	printf("Writing 1st header segment at %d ... ", seg_id);
	if (write_segment(tape_fd, seg_id++, MT_FT_WR_ASYNC, hseg))
		return -1;
	printf("done.\n");
	while (seg_id < scnd) {
		printf("Deleting segment %d ... ", seg_id);
		if (write_segment(tape_fd, seg_id++, MT_FT_WR_DELETE, NULL))
			return -1;
		printf("done\n");
	}
	printf("Writing 2nd header segment at %d ... ", seg_id);
	if (write_segment(tape_fd, seg_id++, MT_FT_WR_ASYNC, hseg))
		return -1;
	printf("done\n");
	while (seg_id < vtbl) {
		printf("Deleting segment %d ... ", seg_id);
		if (write_segment(tape_fd, seg_id++, MT_FT_WR_DELETE, NULL))
			return -1;
		printf("done\n");
	}
	memset(hseg, 0, FT_SEGMENT_SIZE);
	printf("Initializing volume table  at %d ... ", seg_id);
	if (write_segment(tape_fd, vtbl, MT_FT_WR_SINGLE, hseg))
		return -1;
	printf("done.\n");
	return 0;
}


/*
 * Local variables:
 *  version-control: t
 *  kept-new-versions: 5
 *  c-basic-offset: 4
 *  tab-width: 4
 * End:
 */
