// This file is part of Moonlight Creator
//   Copyright (C) 1996-1998  Stephane Rehel
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/*
  Command.cc

  Unix-like command line: one command + some arguments

  Creation: SR, January 30th, 1996
  Revisions:

*/

#include <stdio.h>

#include "Command.h"


/////////////////////////////////////////////////////////////////////////////

void Command::destroy()
{
  while( ! arguments.empty() )
    {
    delete arguments.getFirst();
    arguments.removeFirst();
    }
}

/////////////////////////////////////////////////////////////////////////////

// private
const char* Command::skipSpaces( const char* s ) const
{
  if( s == 0 )
    return 0;

  while( *s == ' ' || *s == '\t' )
    ++s;

  return s;
}

/////////////////////////////////////////////////////////////////////////////

void Command::addArgument( const OString& arg )
{
  assert( arg.assigned() );
  addArgument( arg.get() );
}

/////////////////////////////////////////////////////////////////////////////

void Command::addArgument( const char* arg )
{
  arguments.append( new Argument(arg) );
}

/////////////////////////////////////////////////////////////////////////////

void Command::init( const char* fullCommandLine )
{
  destroy();
  command= "";

  if( fullCommandLine == 0 )
    return;

  register const char* fcl= fullCommandLine;

  for(;;)
    {
    fcl= skipSpaces(fcl);
    if( *fcl == '\0' )
      break;
    register int length= 0;
    while( fcl[length] != '\0' && fcl[length] != ' ' && fcl[length] != '\t' )
      ++length;
    assert( length > 0 );
    addArgument( OString( fcl, length ) );
    fcl += length;
    }

  if( arguments.getNElements() == 0 )
    return;

  command= arguments.getFirst()->arg;
  delete arguments.getFirst();
  arguments.removeFirst();

  error= 0;
  li.reset();
}

/////////////////////////////////////////////////////////////////////////////

void Command::init( int argc, const char* argv[] )
{
  destroy();
  command= "";
  if( argc < 1 || argv == 0 )
    return;

  for( int i= 0; i < argc; ++i )
    {
    const char* s= argv[i];
    if( s == 0 )
      continue;

    addArgument(s);
    }

  if( arguments.getNElements() == 0 )
    return;

  command= arguments.getFirst()->arg;
  delete arguments.getFirst();
  arguments.removeFirst();

  error= 0;
  li.reset();
}

/////////////////////////////////////////////////////////////////////////////

void Command::reset()
{
  li.reset();
  while( ! li.eol() )
    li.next()->used= IFALSE;

  li.reset();
  error= 0;
}

/////////////////////////////////////////////////////////////////////////////

const OString& Command::argv( int i ) const
{
  assert( i >= 0 );
  assert( i < argc() );

  if( i == 0 )
    return command;

  return arguments.get(i)->arg;
}

/////////////////////////////////////////////////////////////////////////////

int Command::getInteger()
{
  if( li.eol() )
    {
    error= 1;
    return 0;
    }

  Argument* a= li.next();
  if( a->used )
    {
    error= 1;
    return 0;
    }

  int i;
  if( sscanf( a->arg.get(), "%d", &i ) != 1 )
    {
    error= 1;
    return 0;
    }

  a->used= ITRUE;

  return i;
}

/////////////////////////////////////////////////////////////////////////////

double Command::getDouble()
{
  if( li.eol() )
    {
    error= 1;
    return 0.;
    }

  Argument* a= li.next();
  if( a->used )
    {
    error= 1;
    return 0.;
    }

  double d;
  if( sscanf( a->arg.get(), "%lf", &d ) != 1 )
    {
    error= 1;
    return 0.;
    }

  a->used= ITRUE;

  return d;
}

/////////////////////////////////////////////////////////////////////////////

const OString& Command::getString()
{
  static OString foo("");

  if( li.eol() )
    {
    error= 1;
    return foo;
    }

  Argument* a= li.next();
  if( a->used )
    {
    error= 1;
    return foo;
    }

  a->used= ITRUE;

  return a->arg;
}

/////////////////////////////////////////////////////////////////////////////

int Command::getIntegerAfter( const char* option )
{
  if( ! findOption(option) )
    {
    error= 1;
    return 0;
    }
  return getInteger();
}

/////////////////////////////////////////////////////////////////////////////

double Command::getDoubleAfter( const char* option )
{
  if( ! findOption(option) )
    {
    error= 1;
    return 0.;
    }
  return getDouble();
}

/////////////////////////////////////////////////////////////////////////////

const OString& Command::getStringAfter( const char* option )
{
  if( ! findOption(option) )
    {
    error= 1;
    static OString foo("");
    return foo;
    }
  return getString();
}

/////////////////////////////////////////////////////////////////////////////

// private
Command::Argument* Command::getFirstArgument()
{
  li.reset();
  return getNextArgument();
}

/////////////////////////////////////////////////////////////////////////////

// private
Command::Argument* Command::getNextArgument()
{
  for(;;)
    {
    if( li.eol() )
      break;
    Argument* a= li.next();
    if( ! a->used )
      return a;
    }

  return 0;
}

/////////////////////////////////////////////////////////////////////////////

const OString& Command::getFirst()
{
  Argument* a= getFirstArgument();

  if( a == 0 )
    {
    static OString foo("");
    return foo;
    }
  a->used= ITRUE;
  return a->arg;
}

/////////////////////////////////////////////////////////////////////////////

const OString& Command::getNext()
{
  Argument* a= getNextArgument();

  if( a == 0 )
    {
    static OString foo("");
    return foo;
    }
  a->used= ITRUE;
  return a->arg;
}

/////////////////////////////////////////////////////////////////////////////

IBOOL Command::findOption( const OString& option )
{
  const OString o= (option[0]=='-') ? option : (OString("-") + option);

  Argument* a= getFirstArgument();
  while( a != 0 )
    {
    if( !a->used && a->option && a->arg.length() >= 2 )
      {
      if( a->arg == o )
        {
        a->used= ITRUE;
        return ITRUE;
        }
      }

    a= getNextArgument();
    }

  return IFALSE;
}

/////////////////////////////////////////////////////////////////////////////

IBOOL Command::findOption( const char* s )
{
  return findOption( OString(s) );
}

/////////////////////////////////////////////////////////////////////////////

IBOOL Command::findOption( char ch )
{
  char s[2];
  s[0]= ch;
  s[1]= '\0';

  return findOption( OString(s) );
}

/////////////////////////////////////////////////////////////////////////////

//
// Example: -I/foo/tools
// -> then getOptionString( 'I' ) returns "/foo/tools"
// return 0 if not found
//
const char* Command::findOptionString( char ch )
{
  Argument* a= getFirstArgument();

  while( a != 0 )
    {
    if( !a->used && a->option && a->arg.length() >= 2 )
      {
      if( a->arg[1] == ch )
        {
        a->used= ITRUE;
        return a->arg.get() + 2;
        }
      }

    a= getNextArgument();
    }

  return 0;
}

/////////////////////////////////////////////////////////////////////////////

const char* Command::getFirstUnusedOption()
{
  Argument* a= getFirstArgument();

  while( a != 0 )
    {
    if( !a->used && a->option && a->arg.length() >= 2 )
      return a->arg.get();

    a= getNextArgument();
    }

  return 0;
}

/////////////////////////////////////////////////////////////////////////////

