// QWeb - An SGML Web Browser
// Copyright (C) 1997  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#ifndef _Browser_h_
#define _Browser_h_

#include <qwidget.h>
#include <qlist.h>
#include "Request.h"

class Canvas;
class History;
class QFrame;
class QLabel;
class QLineEdit;
class QMenuBar;
class QPopupMenu;
class QPushButton;
class SgmlParser;
class TransferLogo;

//: The Browser class is a top-level window for viewing SGML documents.
//. The Browser is the window used for viewing SGML documents and other data
//. types.  Each Browser has a single Canvas that manages a scrolled
//. window containing an appropriate renderer for whatever type of document has
//. been requested.  Usually the renderer is a VerticalRenderer, for displaying
//. an SGML document.
//. <P>
//. The Browser communicates with the global Bookmarks object to provide an
//. up-to-date list of the user's bookmarks in a menu.  The Bookmarks object
//. manages the menu, and processes menu selection events.  The Browser just
//. displays the menu.
//. <P>
//. Each browser has a separate History of visited URLs.  The user can step
//. forward and backward through the History, or go directly to a visited URL
//. by selecting it from the "Go" menu.
//. <P>
//. The user can navigate a "web" of linked documents by clicking on hyperlinks
//. within the document (on text and images), or by typing a URL directly into
//. an entry field and pressing RETURN.
//. <P>
//. A new browser can be spawned from an existing browser, however there is no
//. ownership relationship between the two browsers.
//. <P>
//. The browser also provides menu items, under "Options", for clearing caches,
//. editing style sheets, the SGML catalog, and user preferences, and for
//. saving the SGML catalog and user preferences.
class Browser : public QWidget {
    Q_OBJECT
    QMenuBar*      _menu;
    QPopupMenu*    _file;
    QPopupMenu*    _go;
    QPushButton*   _back;
    QPushButton*   _forward;
    QPushButton*   _home;
    QPushButton*   _reload;
    TransferLogo*  _logo;
    QLineEdit*     _url;
    QPushButton*   _stop;
    QFrame*        _separator1;
    QFrame*        _separator2;
    QLabel*        _status;
    Canvas*        _canvas;
    History*       _history;
    Url            _baseUrl;
    QString        _dtdName;
    int            _statusTimerId;
    bool           _reloadFlag;
protected:
    //. Remove ourself from the list of open browsers, and die.
    void closeEvent( QCloseEvent* e );

    //. Reposition everything and notify the renderers of the size change.
    void resizeEvent( QResizeEvent* e );

    //. Depending on the timer ID... Clear the status bar's text, flip to the
    //. next frame in the cheezy animated icon, or give the renderers a chance
    //. to redraw themselves.
    void timerEvent( QTimerEvent* e );

    //. Notify the Bookmarks that we are the active browser, so that any
    //. selection from the Bookmarks menu will be opened in this browser.
    void enterEvent( QEvent* e );
public:
    //. Create a new browser window.
    Browser( QWidget* parent=0, const char* name = 0 );

    //. Abort all requests, and die.
    ~Browser();

    //. Set the text in the browser window title bar.
    void setTitle( QString title );

    //. Return the current base URL for the document.
    const Url& baseUrl() { return _baseUrl; }

    void setTransfer( bool b );

    bool reload() { return _reloadFlag; }

    void stop();
public slots:
    //. Popup and error dialog, centered over the browser window.
    void errDialog( QString error );

    //. Place the given string in the status bar at the bottom of the browser
    //. window for at most five seconds.
    void status( QString msg );

    //. This function sets the base URL for the browser.  This function is
    //. called when a new document is opened, or when following HTTP redirects.
    void setBaseUrl( const Url& url );

    void urlChanged( const Url& url );

    //. Reload the current page, withut using the cache.
    void doReload();

    //. This is the action routine for the menu item Go-Home and the home
    //. button.  Open the home URL in the browser window, and append it to
    //. the browser history.
    void home();

    //. This is the action routine for the menu item Go-Backward and the back
    //. button.  Open the previous URL in the browser history.
    void back();

    //. This is the action routine for the menu item Go-Forward and the forward
    //. button.  Open the next URL in the browser history.
    void forward();

    //. Take the string in the URL entry field and (attempt to) open it in this
    //. browser.
    void open();

    //. Open the given URL in this browser, and append it to the browser history.
    void open( const Url& url, bool reload=FALSE );

    void startOfData();

    //. This is the action routine for a menu selection event from the Go menu.
    //. Open the curresponding url (if any).
    void goActivated( int id );

    //. If b id TRUE then there is at least one entry in the browser before the
    //. current one.  Activate the back button and the Go-Backward menu item.
    void canGoBack( bool b );
    
    //. If b is TRUE then there is at least one entry in the browser history
    //. after the current one.  Activate the forward button and the Go-Forward
    //. menu item.
    void canGoForward( bool b );

    //. This is the action routine for the menu item File-New Browser...
    //. Create a new top-level browser window, and send it on its way.
    void doNewBrowser();

    //. This is the action routine for the menu item File-Close.  It closes
    //. the browser window, if it can (if it is not the only browser window).
    void doClose();

    //. If b is TRUE then that means there is more than one Browser window open,
    //. and this one can close itself with call exit().  Activate the
    //. File-Close menu item.
    void canClose( bool b );

    //. Asks the Cache to clear all of its stored data.
    void optionsClearDocumentCache();

    //. Asks the DtdManager to clear its cache of DTDs and Style Sheets.
    void optionsClearDtdCache();
    
    //. This is the action routine for the menu item Options-Save Preferences.
    //. It just calls options-save().
    void optionsSavePreferences();

    //. Asks the SgmlCatalog to write itself out to a file.
    void optionsSaveSgmlCatalog();

    //. This function notifies the browser that the SgmlParser has settled on
    //. a document type declaration for the current document.
    void dtdSelected( QString dtdName );

    //. This function is invoked whenever there is a change in the bookamrks.
    //. The old bookmarks menu is deleted, and a new one is requested.
    void bookmarksChanged();

    void doSaveAs();

    void aboutAboutQweb();

    void windowsPreferences();

    void windowsStyleEditor();

    void windowsSgmlCatalog();

    void windowsConsole();
};

#endif
