// QWeb - An SGML Web Browser
// Copyright (C) 1997  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#ifndef _Attribute_h_
#define _Attribute_h_

#include <qstring.h>
#include <qstrlist.h>

//: The Attribute class stores the prototype for an SGML tag attribute.
//. In SGML, all start tags may have one or more attributes associated with
//. them.  These attributes are used to modify the meaning of the element
//. is small ways.  This class is merely a container for an attribute
//. "prototype" that can be used to validate tag attributes, and/or provide
//. default values for tag attributes.
class Attribute {
public:
    enum ValueType {
        Cdata,
        Number,
        Enum,
        Name,
        Names
    };

    enum Type {
        Implied,
        Required,
        Fixed,
        Default
    };
private:
    QString   _name;
    ValueType _valueType;
    Type      _type;
    QStrIList _enums;
    QString   _defaultValue;
public:
    //. Create a new Attribute.  Name is the name of the attribute.  ValueType
    //. is the data type of the value (Cdata, Number, Enum, Name, Names).  Type
    //. is tells whether the value is Implied, Required, Fixed, or has a Default.
    //. Enums is the list of possible values, if the valueType was Enum.
    //. DefaultValue gives the fixed or default value for the attribute.
    Attribute( const QString name,
               ValueType     valueType,
               Type          type,
               QStrIList&    enums,
               QString       defaultValue );

    //. The attribute name.
    const QString  name() { return _name; }

    //. The data type of the attribute value.
    //. <DL>
    //. <DT>Cdata<DD>Raw SGML character data.
    //. <DT>Number<DD>Numeric value (i.e. 5, 2.3, 95%).
    //. <DT>Enum<DD>Enumerated type, only certain string values are allowed.
    //. <DT>Name<DD>I have no idea.
    //. <DT>Names<DD>I have no idea.
    //. </DL>
    ValueType      valueType() { return _valueType; }

    //. Is the attribute value Implied, Required, Fixed, or does it have a Default?
    Type           type() { return _type; }

    //. This is the list of allowed string values if the valueType is Enum.
    QStrIList&     enums() { return _enums; }

    //. This is the default (string) value for the attribute if type is Default.
    const QString  defaultValue() { return _defaultValue; }
};

#endif
