/*
  Copyright (C) 1997  Dimitrios P. Bouras

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   For author contact information, look in the README file.
*/

#include <forms.h>
#include <stdio.h>
#include <stdlib.h>
#include <varargs.h>
#include <strings.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include "xispterm.h"

/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                         Global program storage                          |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

FD_terminal	*fd_terminal;	/* the terminal form */


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                            Utility routines                             |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

/* Print message together with system error message and exit. Note the
   implicit total length of MSGLEN_ERR bytes for the resulting string. */

#define MSGLEN_ERR 128

void doErr(char *msg)
{
	char emsg[MSGLEN_ERR+1];

	if (errno < sys_nerr)
		sprintf(emsg, "xispterm: %s: %s\n", msg, sys_errlist[errno]);
	else
		sprintf(emsg, "xispterm: %s: error #%d\n", msg, errno);
	fputs(emsg, stderr);
	exit(1);
}

/* Search a string for multiple characters returning first occurence */

static char *strfstr(char *haystack, char *needles)
{
	char cn, *hp;

	while ((cn=*needles)) {						/* search for all needles */
		hp = strchr(haystack, cn);				/* found in haystack? */
		if (hp != NULL)							/* yes, return pointer to */
			return(hp);							/* location of matched char */
		++needles;								/* nope, get next needle */
	}
	return(NULL);								/* nothing found */
}

/* Write printf style in the browser object. Note the implicit
   total length of MSGLEN_BROWSER bytes for the resulting string. */

#define MSGLEN_BROWSER (MAXBUF_INPUT*2)

char bmsg[MSGLEN_BROWSER+2] = {0};	/* the string buffer used by bprintf() */
char btmp[MAXBUF_INPUT+1] = {0};	/* temporary buffer for new strings */
char *where = bmsg;					/* incomplete line continuation pointer */

void bDoBuf(void)
{
	char *nl;
	int ll;

	while ((nl= strfstr(bmsg,"\n"))!=NULL) {	/* string contains CR or LF? */
		*nl = 0;								/* yes, mark it as last char */
		ll = fl_get_browser_maxline(			/* get index of last line */
				fd_terminal->lineBrowser);
		fl_replace_browser_line(fd_terminal->	/* display string normally */
			lineBrowser, ll, bmsg);				/* on the browser object */
		fl_addto_browser(fd_terminal->			/* scroll up */
			 lineBrowser, "|");
		strcpy(bmsg, nl+1);						/* move rest to beginning */
	}
}

int bprintf(va_alist) va_dcl
{
	int bw, pending = 0, ll;
	va_list ap;
	char *fmt;
	static int tot = 0;

	va_start(ap);								/* start variable arg list */
	fmt = va_arg(ap, char*);					/* first string is format */
	bw = vsprintf(btmp, fmt, ap);				/* pass the rest to vsprintf */
	va_end(ap);									/* end variable arg list */
	if ((tot+bw) < (MSGLEN_BROWSER-1))			/* do we have space for new? */
		strcat(where, btmp);					/* yup, tack it on the end */
	else {										/* nope, so */
		strcat(where, "\n");					/* end the string here */
		pending = 1;							/* and indicate new pending */
	}
	bDoBuf();									/* process the message buf */
	if (pending) {								/* pending new string? */
		strcpy(bmsg, btmp);						/* yup, copy it in the buffer */
		bDoBuf();								/* process the buffer again */
	}
	tot = strlen(bmsg);							/* total chars so far */
	where = bmsg + tot;							/* pick up from where we left */
	if (tot) {									/* any trailing characters? */
		strcat(bmsg, "|");						/* emulate cursor */
		ll = fl_get_browser_maxline(			/* yup, get last line index */
				fd_terminal->lineBrowser);
		fl_replace_browser_line(fd_terminal->	/* display string normally */
			lineBrowser, ll, bmsg);				/* on the browser object */
		bmsg[tot] = 0;							/* kill cursor */
	}
	return bw;									/* return bytes written */
}

void bKill()
{
	int ll;

	if ( *bmsg ) {								/* if leftover chars exist */
		ll = fl_get_browser_maxline(			/* get index of last line */
				fd_terminal->lineBrowser);
		fl_replace_browser_line(fd_terminal->	/* erase entire line */
			lineBrowser, ll, " ");
		*bmsg = 0;								/* indicate nothing here */
	}
	where = bmsg;								/* and start all over again */
}

void bDel()
{
	int ll;

	if ( *bmsg ) {								/* if leftover chars exist */
		ll = fl_get_browser_maxline(			/* get index of last line */
				fd_terminal->lineBrowser);
		*(bmsg + strlen(bmsg) - 1) = 0;			/* kill last character */
		fl_replace_browser_line(fd_terminal->	/* display string normally */
			lineBrowser, ll, bmsg);				/* on last browser line */
	}
}

/* Reduce colormap usage */

void colorSqueeze(void)
{
	int i;

	for (i=0; i<FL_FREE_COL1; i++) {
		switch (i) {

			case FL_BLACK:						/* except for these which */
			case FL_WHITE:						/* are used in our code */
			case FL_INDIANRED:
			case FL_COL1:
			case FL_RIGHT_BCOL:
			case FL_BOTTOM_BCOL:
			case FL_TOP_BCOL:
			case FL_LEFT_BCOL:
			case FL_MCOL:
				break;

			default:
				fl_set_icm_color(i, 0,0,0);		/* reset all unused internal */
		}										/* colormap colors to black */
	}
}

/* Parse user-specified background color from arguments */

void bgColor(int *argc, char *argv[])
{
	int i, r, g, b;
	char *color;

	for (i=1; i<(*argc-1); i++) {				/* search through args */
		if (!strcmp(argv[i], "-bgcol")) {		/* for "-bgcol" */
			color = (char *)malloc(				/* if found, copy argument */
				strlen(argv[i+1])+1);
			strcpy(color, argv[i+1]);
			for (i+=2; i<*argc; i++)			/* shift all the rest up */
				argv[i-2] = argv[i];			/* "eating up" the "-bgcol" */
			*argc -= 2;
			i = sscanf(color, "#%2X%2X%2X",		/* scan the hex color */
					   &r, &g, &b);
			if (i == 3)							/* if scan successful */
				fl_set_icm_color(FL_INDIANRED,	/* use this color */
					r, g, b);
			else								/* if not */
				fl_set_icm_color(FL_INDIANRED,	/* use the default one */
					0,139,139);
			free(color);						/* free up the space */
			return;
		}
	}
	fl_set_icm_color(FL_INDIANRED, 0,139,139);	/* not found, use default */
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                      Callback routines for xispterm                     |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void doLUpdate(int fd, void *data)
{
	int br;
	char buf[MAXBUF_INPUT+1];

	br = read(fd, &buf, MAXBUF_INPUT);			/* read a buffer full */
	if (br > 0) {								/* if read OK */
		buf[br] = 0;							/* indicate string end */
		bprintf("%s", buf);						/* stick buf in line input */
	}
	else if (br < 0) {							/* read failed */
		if (errno != EAGAIN)					/* stdin input unavailable? */
			doErr("doLUpdate: read");			/* no, abort with diagnostic */
	}
	else										/* EOF on stdin */
		exit(0);								/* good bye :) */
}

int IEvent(FL_OBJECT *obj, int event, FL_Coord mx,
		   FL_Coord my, int key, void *raw_event)
{
	unsigned char buf;

	if (event == FL_KEYBOARD) {					/* is this a keyboard event? */
		buf = key;								/* yup, grab the key */
		switch (key) {
			case 0x08:							/* if Backspace, reflect */
				bDel();							/* change in last line */
				break;

			case 0x15:							/* if ^U */
				bKill();						/* kill last line */
				break;

			default:
				break;
		}
		write(1, &buf, 1);						/* write on stdout */
	}
	return 0;
}

void doTermOK(FL_OBJECT *obj, long param)
{
	fputs("xispterm: done.\n", stderr);			/* notify xisp and */
	exit(0);									/* tell pppd all is OK */
}

void doTermAbort(FL_OBJECT *obj, long param)
{
	fputs("xispterm: ABORT\n", stderr);			/* notify xisp and */
	fputs("xispterm: aborted.\n", stderr);
	exit(1);									/* tell pppd dial failed */
}

/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                                 Main                                    |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

#define MAXLIN_TERM 10 /* experimental; depends on font and browser size */

int main(int argc, char *argv[])
{
	int i;

	fputs("xispterm: started.\n", stderr);
	fcntl(0, F_SETFL, O_NONBLOCK);					/* set non-blocking I/O */
	colorSqueeze();									/* reduce colormap usage */
	bgColor(&argc, argv);							/* parse user bg color */
	fl_initialize(&argc, argv, "X-ISP Terminal",	/* init xforms */
				  NULL, 0);
	fd_terminal = create_form_terminal();			/* create terminal form */
	fl_set_app_mainform(fd_terminal->terminal);		/* this is the main one */
	for (i=0; i<MAXLIN_TERM; i++)					/* fill browser with */
		bprintf("\n");								/* empty lines */
	fl_show_form(fd_terminal->terminal,				/* and show ourself */
				 FL_PLACE_FREE, FL_FULLBORDER,
				 "X-ISP Terminal");

	fl_do_forms();

	return 0;
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |         Program form created with fdesign and annotated by hand         |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

FD_terminal *create_form_terminal(void)
{
  FL_OBJECT *obj;
  FD_terminal *fdui = (FD_terminal *) fl_calloc(1, sizeof(*fdui));

  fdui->terminal = fl_bgn_form(FL_NO_BOX, 322, 251);
  obj = fl_add_box(FL_FLAT_BOX,0,0,322,251,"");
    fl_set_object_color(obj,FL_INDIANRED,FL_COL1);
  fdui->lineBrowser = obj = fl_add_browser(FL_NORMAL_BROWSER,12,13,298,182,"");
    fl_set_object_color(obj,FL_BLACK,FL_YELLOW);
    fl_set_object_lcol(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_ALL);
	fl_set_browser_fontsize(obj, FL_MEDIUM_SIZE);
	fl_set_browser_fontstyle(obj, FL_FIXED_STYLE);
	fl_set_browser_hscrollbar(obj, FL_OFF);
	fl_set_browser_scrollbarsize(obj, 24, 24);

	obj->input = 1;								/* we want input */
	obj->wantkey = FL_KEY_ALL;					/* and all keys sent to us */
	fl_set_object_posthandler(obj, IEvent);		/* register post-handler */
	fl_set_focus_object(fdui->terminal, obj);	/* phocus on this object */

  fdui->termOK = obj = fl_add_button(FL_NORMAL_BUTTON,41,208,81,29,
									 "Continue");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doTermOK,0);
  fdui->termAbort = obj = fl_add_button(FL_NORMAL_BUTTON,199,208,81,29,
										"Abort");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doTermAbort,0);

    fl_add_io_callback(0,FL_READ, doLUpdate, NULL);	/* register I/O callback */

  fl_end_form();

  fdui->terminal->fdui = fdui;

  return fdui;
}

