/*
 *	pop3d		- IP/TCP/POP3 server for UNIX 4.3BSD
 *			  Post Office Protocol - Version 3 (RFC1225)
 *
 *      (C) Copyright 1991 Regents of the University of California
 *
 *      Permission to use, copy, modify, and distribute this program
 *      for any purpose and without fee is hereby granted, provided
 *      that this copyright and permission notice appear on all copies
 *      and supporting documentation, the name of University of California
 *      not be used in advertising or publicity pertaining to distribution
 *      of the program without specific prior permission, and notice be
 *      given in supporting documentation that copying and distribution is
 *      by permission of the University of California.
 *      The University of California makes no representations about
 *      the suitability of this software for any purpose.  It is provided
 *      "as is" without express or implied warranty.
 *
 *	Katie Stevens
 *	dkstevens@ucdavis.edu
 * 	Information Technology -- Campus Access Point
 *	University of California, Davis
 *
 **************************************
 *
 *	main.c
 *
 *	REVISIONS:
 *		02-27-90 [ks]	original implementation
 *	1.000	03-04-90 [ks]
 *	1.001	06-24-90 [ks]	implement optional TOP command
 *	1.002	07-22-91 [ks]	-- reset index counter after folder rewind
 *				   in fld_release (Thanks to John Briggs,
 *				   vaxs09@vitro.com, Vitro Corporation,
 *				   Silver Spring, MD for finding this bug!)
 *				-- set umask() value explicitly (Thanks to
 *				   Vikas Aggarwal, JvNCnet, Princeton, NJ
 *				   for suggesting this)
 *	1.003	03-92    [ks]   close mailbox before return from main()
 *	1.004   11-13-91 [ks]	leave original mailbox intact during POP
 *				session (Thanks to Dave Cooley,
 *				dwcooley@colby.edu for suggesting this)
 *		08-93	 [ks]	Thanks to Kurt Jordan, Purdue Calumet
 *				for the port to SVR3.2
 *	1.005	06-17-94 [ks]	port to SVR4/ANSIC; fold BSD, SVR3.2, SVR4
 *				into one source ( Thanks to Mark Horstman,
 *				Southwester Bell Telephone for this work )
 *	1.006	07-20-94 [ks]   add hooks for syslog tracking of POP
 *                              sessions; conditional compilation ( Thanks
 *				to David Wong, UC Davis for this work )
 *	1.007	07-29-94 [ks]	add '\r' to POP server greeting string
 */

#include <stdio.h>
#include <signal.h>
#include <ctype.h>
#ifdef SYSV
#include <stropts.h>
#endif /* SYSV */
#include <sys/param.h>
#ifdef TLI
#include <sys/ioctl.h>
#endif /* TLI */
#ifdef SVR4
#include <netdb.h>
#endif /* SVR4 */
#include "pop3.h"

#ifdef LOG_SESSIONS
#include <syslog.h>
#define FACILITY	LOG_AUTH
#endif

#define VERSION		"1.007"
#define REVDATE		"July, 1994"
#define MAX_ATTEMPTS	3			/* maximum invalid logins */

char *svr_hostname;				/* Hostname of POP3 server */
char svr_buf[SVR_BUFSIZ+2];			/* Buffer for server I/O */
char cli_user[CLI_BUFSIZ];			/* Save client username */
int mcounter=0;
int bcounter=0;
int nLoginAttempts=0;

static char *svr_invalid = "-ERR Invalid command; valid commands:";

#ifdef DEBUG
FILE *logfp = NULL;				/* File for recording session */
#endif

/* Routines in this file */
#ifdef ANSIC
static void initialize(void);
static void svr_timeout(int);
static void int_hangup(int);
static void int_progerr(int);

static int svr_auth(int, char *);
static int svr_pass(int, char *);
static int svr_trans(int, char *);
static int svr_fold(int, char *);
static int svr_shutdown(void);
#else /* !ANSIC */
static void initialize();
static void svr_timeout();
static void int_hangup();
static void int_progerr();

static int svr_auth();
static int svr_pass();
static int svr_trans();
static int svr_fold();
static int svr_shutdown();
#endif /* ANSIC */

/**************************************************************************/

/* Initialize POP3 server */
static void
#ifdef ANSIC
initialize(void)
#else /* !ANSIC */
initialize()
#endif /* ANSIC */
{
	char buf[MAXHOSTNAMELEN+1];

	/* File permissions for owner only */
	umask(077);		/* [1.002] */

#ifdef DEBUG
	/* Prepare log file */
	logfp = fopen(LOGFILE,"w");
	fprintf(logfp,"POP3 server startup; version %s (%s)\n",
		VERSION,REVDATE);
#endif
	/* Get our hostname */
	gethostname(buf,MAXHOSTNAMELEN);
#ifdef TLI
	if (ioctl(fileno(stdin), I_PUSH, "tirdwr") == -1 ) {
	    perror("ioctl");
	    exit(7);
	}
#endif /* TLI */
	svr_hostname = malloc(strlen(buf) + 1);
	if (svr_hostname == NULL)
		fail(FAIL_OUT_OF_MEMORY);
	strcpy(svr_hostname,buf);

	/* Handle process signals ourself */
	signal(SIGALRM, svr_timeout);		/* timer expiration */

	signal(SIGHUP, int_hangup);		/* socket signals */
	signal(SIGPIPE, int_hangup);
	signal(SIGURG, int_hangup);
	signal(SIGTERM, int_hangup);

	signal(SIGBUS, int_progerr);		/* fatal program errors */
	signal(SIGSEGV, int_progerr);
	signal(SIGILL, int_progerr);
	signal(SIGIOT, int_progerr);
}

/* Timeout while waiting for next client command */
static void
#ifdef ANSIC
svr_timeout(int sig)
#else /* !ANSIC */
svr_timeout(sig)
int sig;
#endif /* ANSIC */
{
	fld_release();				/* Release mailbox folder */
	fail(FAIL_LOST_CLIENT);			/* Exit POP3 server */
}
/* Timeout while waiting for next client command */
static void
#ifdef ANSIC
int_hangup(int sig)
#else /* !ANSIC */
int_hangup(sig)
int sig;
#endif /* ANSIC */
{
	fld_release();				/* Release mailbox folder */
	fail(FAIL_HANGUP);			/* Exit POP3 server */
}
/* Timeout while waiting for next client command */
static void
#ifdef ANSIC
int_progerr(int sig)
#else /* !ANSIC */
int_progerr(sig)
int sig;
#endif /* ANSIC */
{
	fld_release();				/* Release mailbox folder */
	fail(FAIL_PROGERR);			/* Exit POP3 server */
}

/**************************************************************************/

/* Server Authentification State; process client USER command */
static int
#ifdef ANSIC
svr_auth(int state, char *inbuf)
#else /* !ANSIC */
svr_auth(state,inbuf)
int state;
char *inbuf;
#endif /* ANSIC */
{
	if (strncmp(inbuf,"quit",4) == 0)
		return(svr_shutdown());
	/* Expecting USER command */
	if (strncmp(inbuf,"user",4) == 0) {
		inbuf += 4;
		EATSPACE(inbuf);
		strcpy(cli_user,inbuf);
		strcpy(svr_buf,"+OK please send PASS command\r\n");
		state = SVR_PASS_STATE;
	} else {
		sprintf(svr_buf,"%s  USER,  QUIT\r\n",svr_invalid);
	}
	return(state);
}

/* Server Password State; process client PASS command */
static int
#ifdef ANSIC
svr_pass(int state, char *inbuf)
#else /* !ANSIC */
svr_pass(state,inbuf)
int state;
char *inbuf;
#endif /* ANSIC */
{
	int cnt;

	if (strncmp(inbuf,"quit",4) == 0)
		return(svr_shutdown());
	/* Expecting PASS command */
	if (strncmp(inbuf,"pass",4) != 0) {
		sprintf(svr_buf,"%s  PASS,  QUIT\r\n",svr_invalid);
		return(state);
	}
	/* Verify usercode/password pair */
	inbuf += 4;
	EATSPACE(inbuf);
	if (verify_user(cli_user,inbuf) == -1) {
		nLoginAttempts++;
#ifdef LOG_SESSIONS
		if ( nLoginAttempts == 1 ) {
			syslog(LOG_WARNING,"%d LOGIN FAILURE, %s\n", nLoginAttempts, cli_user);
		} else {
			syslog(LOG_WARNING,"%d LOGIN FAILURES, %s\n", nLoginAttempts, cli_user);
		}
#endif
		if ( nLoginAttempts >= MAX_ATTEMPTS ) {
			return(svr_shutdown());
		}
		strcpy(svr_buf,"-ERR invalid usercode or password, please try again\r\n");
		return(SVR_AUTH_STATE);
	} else {
		nLoginAttempts=0;
	}

	strcpy(svr_buf,DEF_MAIL_DIR);
	strcat(svr_buf,cli_user);
	return(fld_fromsp(svr_buf));
}

/* Server Transaction State; process client mailbox command */
static int
#ifdef ANSIC
svr_trans(int state, char *inbuf)
#else /* !ANSIC */
svr_trans(state,inbuf)
int state;
char *inbuf;
#endif /* ANSIC */
{
	int msgnum;

	if (strncmp(inbuf,"quit",4) == 0)
		return(svr_shutdown());
	/* Expecting mailbox command */
	if (strncmp(inbuf,"dele",4) == 0) {
		inbuf += 4;
		EATSPACE(inbuf);
		if (*inbuf == NULL_CHAR)
			sprintf(svr_buf,"-ERR message number required (e.g.  DELE 1)\r\n");
		else
			fld_delete(atoi(inbuf));
	} else if (strncmp(inbuf,"host",4) == 0) {
		inbuf += 4;
		EATSPACE(inbuf);
		if (*inbuf == NULL_CHAR)
			sprintf(svr_buf,"-ERR must specify hostname\r\n");
		else
			state = fld_bsmtp(inbuf);
	} else if (strncmp(inbuf,"last",4) == 0) {
		fld_last();
	} else if (strncmp(inbuf,"list",4) == 0) {
		inbuf += 4;
		EATSPACE(inbuf);
		if (*inbuf == NULL_CHAR)
			fld_list(-1);
		else
			fld_list(atoi(inbuf));
#ifdef MBOX_CMD
	} else if (strncmp(inbuf,"mbox",4) == 0) {
		inbuf += 4;
		EATSPACE(inbuf);
		if (*inbuf == NULL_CHAR)
			sprintf(svr_buf,"-ERR must specify mailbox filename\r\n");
		else
			state = fld_fromsp(inbuf);
#endif
	} else if (strncmp(inbuf,"noop",4) == 0) {
		strcpy(svr_buf,"+OK\r\n");
	} else if (strncmp(inbuf,"retr",4) == 0) {
		inbuf += 4;
		EATSPACE(inbuf);
		if (*inbuf == NULL_CHAR) {
			sprintf(svr_buf,"-ERR message number required (e.g.  RETR 1)\r\n");
		} else
			fld_retr(atoi(inbuf),-1);
	} else if (strncmp(inbuf,"rset",4) == 0) {
		fld_reset();
	} else if (strncmp(inbuf,"stat",4) == 0) {
		fld_stat();
	} else if (strncmp(inbuf,"top",3) == 0) {
		inbuf += 3;
		EATSPACE(inbuf);
		if (*inbuf == NULL_CHAR) {
			sprintf(svr_buf,"-ERR message number and line count required (e.g.  TOP 1 7)\r\n");
		} else {
			msgnum = atoi(inbuf);
			while (!isspace(*inbuf)) ++inbuf;
			EATSPACE(inbuf);
			if (*inbuf == NULL_CHAR)
				sprintf(svr_buf,"-ERR line count required (e.g.  TOP 1 7)\r\n");
			else
				fld_retr(msgnum,atoi(inbuf));
		}
	} else {
		sprintf(svr_buf,
			"%s  DELE, HOST, LAST, LIST,"
#ifdef MBOX_CMD
			" MBOX,"
#endif
			" NOOP, RETR, RSET, STAT, TOP  or  QUIT\r\n",
			svr_invalid);
	}
	return(state);
}

/* Server Folder State; need to open another folder */
static int
#ifdef ANSIC
svr_fold(int state, char *inbuf)
#else /* !ANSIC */
svr_fold(state,inbuf)
int state;
char *inbuf;
#endif /* ANSIC */
{
	if (strncmp(inbuf,"quit",4) == 0)
		return(svr_shutdown());
	if (strncmp(inbuf,"host",4) == 0) {
		inbuf += 4;
		EATSPACE(inbuf);
		state = fld_bsmtp(inbuf);
#ifdef MBOX_CMD
	} else if (strncmp(inbuf,"mbox",4) == 0) {
		inbuf += 4;
		EATSPACE(inbuf);
		state = fld_fromsp(inbuf);
#endif
	} else if (strncmp(inbuf,"noop",4) == 0) {
		strcpy(svr_buf,"+OK\r\n");
	} else {
		sprintf(svr_buf,"%s  HOST,  MBOX,  NOOP  or  QUIT\r\n",svr_invalid);
	}
	return(state);
}

/* Prepare to shutdown POP3 server */
static int
#ifdef ANSIC
svr_shutdown(void)
#else /* !ANSIC */
svr_shutdown()
#endif /* ANSIC */
{
	fld_release();
	sprintf(svr_buf,"+OK %s POP3 Server (Version %s) shutdown.\r\n",
		svr_hostname,VERSION);
	return(SVR_DONE_STATE);
}

/**************************************/

void
#ifdef ANSIC
svr_data_out( char *buf)
#else /* !ANSIC */
svr_data_out(buf)
char *buf;
#endif /* ANSIC */
{
	/* Send out response to client */
	alarm(SVR_TIMEOUT_SEND);
	fputs(buf,stdout);
	fflush(stdout);
	alarm(0);
}

/**************************************************************************/

int
#ifdef ANSIC
main(int argc, char *argv[])
#else /* !ANSIC */
main(argc, argv)
int argc;
char *argv[];
#endif /* ANSIC */
{
	int svr_state = SVR_LISTEN_STATE;	/* State of POP3 server */
	char cli_buf[CLI_BUFSIZ];		/* Buffer for client cmds */

	initialize();


#ifdef LOG_SESSIONS
	openlog( "pop3d", LOG_PID, FACILITY );
#endif
	fprintf(stdout,"+OK %s POP3 Server (Version %s) ready.\r\n",
		svr_hostname,VERSION);
	fflush(stdout);
	svr_state = SVR_AUTH_STATE;
	for ( ; ; ) {
		/* Wait for command from client */
		alarm(SVR_TIMEOUT_CLI);
		if (fgetl(cli_buf,CLI_BUFSIZ,stdin) == NULL)
			break;
		alarm(0);

		/* Take action on client command */
		cmd_prepare(cli_buf);
#ifdef DEBUG
		if ((cli_buf[0] == 'p')||(cli_buf[0] == 'P'))
			fprintf(logfp,"CLI: PASS\n",cli_buf);
		else
			fprintf(logfp,"CLI: %s\n",cli_buf);
#endif
		switch(svr_state) {
		case SVR_AUTH_STATE:	/* Expecting USER command */
			svr_state = svr_auth(svr_state,cli_buf);
			break;
		case SVR_PASS_STATE:	/* Expecting PASS command */
			svr_state = svr_pass(svr_state,cli_buf);
			break;
		case SVR_TRANS_STATE:	/* Expecting mailbox command */
			svr_state = svr_trans(svr_state,cli_buf);
			break;
		case SVR_FOLD_STATE:	/* Need to open another mailbox */
			svr_state = svr_fold(svr_state,cli_buf);
			break;
		default:
			fail(FAIL_CONFUSION);		/* Wont return */
			break;
		}
#ifdef DEBUG
		fprintf(logfp,"SVR: %s",svr_buf);
#endif

		/* Send out response to client */
		alarm(SVR_TIMEOUT_SEND);
		fputs(svr_buf,stdout);
		fflush(stdout);
		alarm(0);
		if (ferror(stdout))
			break;

		/* Exit when server has sent goodbye */
		if (svr_state == SVR_DONE_STATE)
			break;
	}
	fld_release();		/* [1.003] Make sure folder is released */
#ifdef LOG_SESSIONS
	syslog( LOG_INFO | FACILITY, "User: %s (%d messages, %d octets retrieved)", cli_user, mcounter, bcounter);
	closelog();
#endif
#ifdef TLI
	if (ioctl(fileno(stdin), I_POP, 0) == -1 ) {
	    perror("ioctl");
	    exit(7);
	}
#endif /* TLI */
	exit(0);
}
