/*
   Time-stamp: <95/12/07 00:50:34 yusuf>
*/


#include "taper.h"

void paint_main();		       /* in taper.c */
inline _errstat is_regfile(int d)
{
/* Returns 1 if dv is the file handle of a regular file */
    struct stat sb;
    
    fstat(d, &sb);
    return S_ISREG(sb.st_mode);
}


_errstat tape_fsf(int count)
{
    struct mtop m;
    char   l[200];
    
    m.mt_op = MTFSF;
    m.mt_count = count;

/* When doing fsf, must flush buffers */    
    if (write_offset != -1)
      flush_buffers();
    read_offset = -1;				 /* discard rest of block */
    if (!have_fsf) return 0;
    sprintf(l, "Advancing %d volumes", count);
    if (log_level > 2) write_log(l);
    return (is_regfile(dv)) ? lseek(dv, 0L, SEEK_END) : 
                              ioctl(dv, MTIOCTOP, (char *) &m);
}


_errstat tape_readheader(struct tape_header *t, _s8 allowz)
{
/* Reads the tape header. It assumes that the tape is at the
 * beginning and no data has been read in as yet.
 * 
 * Any leading zeroes are ignored. If there is a whole block of
 * zeroes (actually, the last sizeof(struct tape_header) bytes
 *	   are not checked), this is assumed to be an error.
 * 
 * We also assume that a whole block must be read in - if this
 * is the tape header, we can't be at the end of the tape so
 * we won't accept partial block reads
 * 
 * If allowz is 1, then if it can't read any bytes from the
 * tape, it returns (setting the tape_header structure to 0)
 * otherwise, this is an error

 * Returns 0 if read OK, -1 if error
 */
    char s[100];
    char *m;
    struct mtop mt;
    _errstat done_it=0;
    
    while (1) {
	memset(t, 0, sizeof(*t));
	if (log_level > 2) write_log("Reading in block for tape header");
	read_buffer_count = read(dv, read_buffer, block_size);
	blocks_passed=1;
	if (read_buffer_count != block_size) {	 /* read in a block */
	    if (tape_type == TAPE_TYPE_ZFTAPE) { /* zftape has a problem sometimes, so read again to double check */
		if (log_level > 2) write_log("Invalid read count - trying to re-set block size");
		if ((read_buffer_count == -1) && (errno == EINVAL)) {   /* correct block size initially */
		    if (tape_get_blk_size(dv) == -1) return (allowz) ? 0 : -1;
		    if (log_level > 2) write_log("Rereading tape header block");
		    read_buffer_count = read(dv, read_buffer, block_size);
		    if (read_buffer_count != block_size) {
			if (allowz) return 0;	 /* errors allowed */
			write_fatal_log("reading in tape header block");
			return -1;
		    }
		    goto success_read;
		}
		if (allowz) return 0;		 /* allowed to have nothing */
	    }
	    else {
		write_fatal_log("reading in tape header block");
		return -1;
	    }
	}
	success_read:;
	read_offset = 0;
	
	if (log_level > 2) write_log("Skipping leading zeroes");
	while ((*(read_buffer+read_offset) == 0) && 
	       (read_offset < (block_size-sizeof(struct tape_header))))
	  read_offset++;				 /* look for first non-zero byte */
	if (read_offset == block_size-sizeof(struct tape_header))  {
	    if (allowz) return 0;			 /* allowed to have nothing */
	    write_log("Too many leading zeroes while reading in tape header");
	    log_errors++;
	    return -1;				 /* error - full block of zeroes */
	}
	if (log_level > 2) {
	    sprintf(s, "Found %d leading zeroes", read_offset);
	    write_log(s);
	}
	*t = *((struct tape_header *) (read_buffer+read_offset));
	if (log_level > 2) {
	    sprintf(s, "Read in tape header %x %u %d", t->magic, t->archive_id, t->tape_number);
	    write_log(s);
	}
	read_offset += sizeof(struct tape_header);
	if ((t->magic == TAPER_MAGIC_NUMBER) || (t->magic == TAPER_4_MAGIC_NUMBER))
	  break;				 /* found magics - must be OK */
	if (tape_type == TAPE_TYPE_ZFTAPE) {	 /* zftape sometimes misses first read so try again */
	    if (!is_regfile(dv) && !done_it) {	 /* only if using tape drive */
		mt.mt_op = MTREW;		 /* try reading more than one segment  */
		mt.mt_count = 1;
		ioctl(dv, MTIOCTOP, (char *) &mt);   /* rewind */
		m = malloc((28672/block_size+1)*block_size);
		if (m == NULL)
		  do_exit(ERROR_MEMORY);
		sprintf(s, "Special zftape. Reading %d, Read %d",((28672/block_size+1)*block_size),
			read(dv, m, (28672/block_size+1)*block_size));   /* read more than 28K */
		if (log_level > 2) write_log(s);
		free(m);
		mt.mt_op = MTREW;
		mt.mt_count = 1;
		sprintf(s, "Special zftape. Rewinding %d",/* rewind */
			ioctl (dv, MTIOCTOP, (char *) &mt));	
		if (log_level > 2) write_log(s);
		fsync(dv);
		close(dv);			 /* by close */
		if (tape_open(O_RDWR)  == -1) /* and reopen */
		  return -1;
		done_it = 1;			 /* only do it once */
	    }
	}
	else					 /* regular file or already done it - no problems */
	  break;
    }
    return 0;
}

						 
_errstat check_tape(WINDOW *mes, int line, _s32 tape_required)
{
/* Prompts user for tape_required tape on archive in tdh.archive_id
   Also makes sure correct archive id is in tape  
 
   Returns -1 if error or user aborted
   Returns 1 if OK
*/

    char   l[200];
    struct tape_header tdh1;
    WINDOW *mymes=NULL;

    if ( (tdh.archive_id == ifd.archive_id) &&
         (tdh.tape_number == tape_required) ) return 1; /* correct tape already read */
    if (mf) {					 /* doing unattended */
	strcpy(l, "Wrong tape in drive. Backup ABORTED\n");
	write(mf, l, strlen(l));
	return -1;
    }
    sprintf(l, "Prompting for insertion of tape %d", tape_required);
    if (log_level > 2) write_log(l);
    sprintf(l, "Please insert tape %d of archive %u in tape drive", tape_required, ifd.archive_id);
    if (!message_box(l, MB_OKCANCEL)) {
	if (lf)
	  write_log("ERROR: User aborted while prompting for new tape");
        return -1;
    }
    paint_main();
    if (mes==NULL) {
	mymes = status_box(mymes, "", 1, TRUE, 1);
	line = 1;
    }
    else
      mymes = mes;
    while (1) {
	mymes = status_box(mymes, "Identifying tape", line, FALSE, 1);
	tape_close();
	sprintf(l, "Opening tape to ensure that tape #%d of archive %d is in drive", 
		tape_required, tdh.archive_id);
	if (log_level > 2) write_log(l);
	if (tape_rewind() == -1) return -1;
	if (tape_open(O_RDWR) == -1) {
	    write_error_log(l);
	    if (mes==NULL)
	      close_statusbox(mymes);
	    return -1;
	}
	vols_passed = 0;			 /* no volumes passed on this tape */
	tdh1.magic = 0;
	tape_readheader(&tdh1, 0);
	if (tdh1.magic != TAPER_MAGIC_NUMBER) {
	    if (!message_box("This tape doesn't contain taper data", MB_OKCANCEL))
		return -1;
	    paint_main();
	    continue;
	}
	if (tdh1.archive_id != tdh.archive_id) {
	    sprintf(l, "This tape isn't part of archive %s", tdh.archive_title);
	    if (!message_box(l, MB_OKCANCEL))
	      return -1;
	    paint_main();
	    continue;
	}
	if (tdh1.tape_number != tape_required) {
	    sprintf(l, "This is tape %d. I require %d", tdh1.tape_number, tape_required);
	    if (!message_box(l, MB_OKCANCEL))
	      return -1;
	    paint_main();
	    continue;
	}
	break;
    }
    if (mes==NULL)
      close_statusbox(mymes);
    else
      status_box(mes, "", line, FALSE, 1);
    paint_main();
    tdh = tdh1;
    return 1;
}



_errstat tape_rewind()
{
/* Rewinds tape. Tape must be closed prior to calling of rewind.
 * Returns 0 if rewound, -1 if error

   If allow_err = 1, then no error message is printed (error code still returned)
 * 
 * It doesn't use tape_open/tape_close to avoid buffering problems
*/
    _errstat err=0, myerrno, x;
    struct mtop m;
    
    m.mt_count = 1;
    m.mt_op = MTREW;
    
    if (!have_rewind) return 0;
    if (log_level > 2) write_log("Rewinding tape");
    if ((x=open(tape, O_RDWR)) == -1)		 /* try and open tape */
      return do_exit(ERROR_OPENING_BACKUP);
    if (!is_regfile(x))
      err = ioctl (x, MTIOCTOP, (char *) &m);	 /* do status if not regular file */
    myerrno = errno;				 /* preserve errno accross close */
    close(x);					 /* close */
    errno = myerrno;
    return (err == -1) ? do_exit(ERROR_REWIND) : 0;
}


void tape_set_blk_size()
{
/* Sets the block size, if required */    
    struct mtop m;
    _errstat err;
    char l[100];

    block_size = org_block_size;
    m.mt_op = MTSETBLK;
    m.mt_count = block_size;

    if (set_blksize) {
	sprintf(l, "Setting block size to %d", block_size);
	if (log_level > 2) write_log(l);
	err = (is_regfile(dv)) ? 0 : ioctl(dv, MTIOCTOP, (char *) &m);
	if (err == -1) do_exit(ERROR_SETTING_BLKSIZE);
	if (init_buffers(1) == -1) do_exit(ERROR_SETTING_BLKSIZE);/* reallocate buffer sizes */
    }
}

_errstat tape_get_blk_size(_s32 x)
{
    struct mtblksz m;
    _errstat err;
    char s[100];

    if (get_blksize) {
	if (log_level > 2) write_log("Getting block size");
	m.mt_blksz = block_size;
	err = (is_regfile(x)) ? 0 : ioctl(x, MTIOC_ZFTAPE_GETBLKSZ, (char *) &m);
	if (err == -1) 
	  do_exit(ERROR_GETTING_BLKSIZE);
	else
	  block_size = m.mt_blksz;
	sprintf(s, "Got block size %d", block_size);
	if (log_level > 2) write_log(s);
	if (init_buffers(1) == -1)		       /* reallocate buffer sizes */
	  return -1;
    }
    return 0;
}


_errstat tape_erase()
{
    struct mtop m;
    _errstat err;

    tape_set_blk_size();
    if (!erase_tape) {
	if (log_level > 2) write_log("Not erasing tape");
	return 0;					 /* don't need to erase tape */
    }

    m.mt_op = MTERASE;
    m.mt_count = 1;
    
    if (log_level > 2) write_log("Erasing tape");
    err = (is_regfile(dv)) ? 0 : ioctl(dv, MTIOCTOP, (char *) &m);
    if (err == -1) return do_exit(ERROR_ERASING_TAPE);
    return 0;
}

void clear_buffer_pointers()
{
    left_tr = 0;
    read_offset = -1; 
    read_buffer_count = 0;
    write_offset = 0;
    w_cur_pos = 0;
    w_current_buffer = w_buffer_1;
    write_buf_no = 0;
}


_errstat mount_device()
{
/* If a mounted type, tries to mount the device.
 * 
 * Returns -1 if error, 0 otherwise */
    if (tape_type != TAPE_TYPE_REMOVABLE) return 0;
    return 0;
}


_errstat ummount_device()
{
/* Tries to umount device */
    if (tape_type != TAPE_TYPE_REMOVABLE) return 0;
    return 0;
}


_errstat ntape_open(int mode)
{
/* Opens non-rewinding device. 
 
   If allower == 1, allows errors, otherwise error message */
    char l[200];

    if (!have_rewind) return tape_open(mode);
    if (mount_device() == -1) return -1;
    sprintf(l, "Opening non-rewinding %s", ntape);
    if (log_level > 2) write_log(l);
    clear_buffer_pointers();
    dv = open(ntape, mode);
    if (dv == -1) 
	return do_exit(ERROR_OPENING_BACKUP);
    if (tape_get_blk_size(dv) == -1) return -1;
    return dv;
}


_errstat tape_open_engine(int mode, int rezero)
{
/* Opens rewinding device */
    char l[200];

    if (mount_device() == -1) return -1;
    sprintf(l, "Opening rewinding %s", tape);
    if (log_level > 2) write_log(l);
    if (rezero) clear_buffer_pointers();	 /* only if user wants us to */
    dv = open(tape, mode);
    if (dv == -1) 
	return do_exit(ERROR_OPENING_BACKUP);
    if (tape_get_blk_size(dv) == -1) return -1;
    return dv;
}


_errstat tape_open(int mode)
{
/* Opens rewinding device */
    return tape_open_engine(mode, 1);
}


_errstat tape_close_engine(_s8 flush)
{
    volatile char *x;
    _errstat ret;

    /* closes tape device */
    if (flush) {
	flush_buffers();
	x = write_buf_no;
	if ((x != NULL) && (x != (char *) -1)) {
	    if (log_level > 2) write_log("Close - waiting for child to finish writing");
	    while ((x != NULL) && (x != (char *) -1))
	      x = write_buf_no;
	}
	read_offset = -1;
	if (x == (char *) -1)
	  do_exit(ERROR_WRITING);
    }
    if (write_pid) {
	kill(write_pid, SIGTERM);
	if (log_level > 3) write_log("Close - killed child done");
	waitpid(write_pid, NULL, 0);
	if (log_level > 3) write_log("Close - waitpid child done");
	write_pid = 0;
    }
    fsync(dv);
    if (log_level > 2) write_log("Closing tape");
    ret = (dv) ? close(dv) : 0;
    dv = -1;
    if (ummount_device() == -1) return -1;
    return ret;
}


_errstat tape_close()
{
    return tape_close_engine(1);
}


int read_from_buf(char *buf, size_t count)
{
    size_t tr_bytes, fit_in;
    
    if (read_offset == -1) {			 /* nothing in buffer */
	if (log_level > 2) write_log("Reading in block from tape");
	read_buffer_count = read(dv, read_buffer, block_size);   /* read in a block */
	blocks_passed++;
	read_offset = 0;			 /* set offset to zero */
    }
    if (read_buffer_count == -1) return -1;	 /* problem reading in buffer */
    if (read_buffer_count == 0) return 0;	 /* no more data in buffer */
    tr_bytes = 0;				 /* either now or previously */
    while (tr_bytes < count) {			 /* transfer bytes to user buffer */
	fit_in = min(count-tr_bytes, read_buffer_count-read_offset);
	memcpy(buf+tr_bytes, read_buffer+read_offset, fit_in);
	tr_bytes += fit_in;
	read_offset += fit_in;
	if (read_offset >= read_buffer_count) {	 /* run out of bytes in buffer */
	    if (read_buffer_count < block_size)	 /* means that last time, didn't get all that */
	      return tr_bytes;			 /* we wanted - ie. end of tape */
	    if (log_level > 2) write_log("Reading in block from tape");
	    read_buffer_count = read(dv, read_buffer, block_size);
	    blocks_passed++;
	    if (read_buffer_count < 1) return tr_bytes;   /* if couldn't get any more, accept what we got */
	    read_offset = 0;
	}
    }
    return tr_bytes;
}


_errstat write_tape_header(struct tape_header *tdh)
{
/* Writes the tape header. Assumes that the tape is at the 
 * beginning. Also assumes that there may be valid data waiting
 * to be written in the write buffer
 * 
 * Because we should be at the beginning of the tape, incomplete
 * writes are not tolerated
 * 
 * Returns 0 if OK, -1 if error
 */
    char *my_write_buffer;
    _s32  oldwo;
    
    my_write_buffer = alloca(block_size);
    if (my_write_buffer == NULL)
      do_exit(ERROR_MEMORY);
    if (log_level > 2) write_log("Writing tape header");
    if (write_offset + sizeof(struct tape_header) < block_size) {
	if (log_level > 2) write_log("Just adding tape header to front of write buffer");
	memcpy(my_write_buffer+sizeof(struct tape_header),   /* room in buffer */
	       write_buffer, write_offset);	 /* for the tape header */
	*(struct tape_header *) my_write_buffer = *tdh;
	write_offset += sizeof(struct tape_header);
	memcpy(write_buffer, my_write_buffer, write_offset);
	return 0;
    }

/* No room in write buffer for tape header - therefore, write out
 * the tape header plus what we can of write buffer and then remove
 * what's been written from write buffer
 */
    if (log_level > 2) write_log("No room in buffer for tape header");
    *(struct tape_header *) my_write_buffer = *tdh;
    memcpy(my_write_buffer+sizeof(struct tape_header),
	   write_buffer, block_size - sizeof(struct tape_header));
    if (write(dv, my_write_buffer, block_size) < block_size) {
	write_fatal_log("writing tape header block");
	my_free(my_write_buffer);
	return -1;
    }
    blocks_passed++;
    my_free(my_write_buffer);
    oldwo = write_offset;
    write_offset -= (block_size - sizeof(struct tape_header));
    if (write_offset)
      memcpy(write_buffer, write_buffer+oldwo-write_offset, write_offset);
    return 0;
}


_errstat do_write_block(char *buf_to_write, _s32 old_length, _s8 fc)
{
/* Writes the buffer at buf_to_write of old_length out to
 * the tape. If fc, then will write out the last partial
 * block, otherwise buffers it
 * 
 * If runs out of space, sets left_tr to how many bytes it wrote
 * and returns -2
 * 
 * Returns -1 if error, 0 otherwise. 
 
 */
    size_t c, c1;
    _s32  tr;
    char  s[100];
    
    tr = 0;					 /* transferred 0 */
    while (tr <= old_length) {
	if (old_length-tr+write_offset < block_size) {
	    memcpy(write_buffer+write_offset, buf_to_write+tr,   /* copy remaining */
		   old_length-tr);   /* data to write buffer */
	    memset(write_buffer+old_length-tr+write_offset, 0, block_size-
		   (old_length-tr+write_offset));   /* pad out rest with zeroes */
	    if (fc) {				 
		if (log_level > 3) 		 /* flushing for close */
		  write_log("W: half block being written out");
	    }
	    else {				 /* not flushing */
		write_offset += (old_length-tr);
		if (log_level > 3) {
		    sprintf(s, "W: half block buffered %d", write_offset);
		    write_log(s);
		}
		left_tr = 0;
		return 0;			 /* not tripleb buffering */
	    }
	}
	else 
	  memcpy(write_buffer+write_offset, buf_to_write+tr, 
		 block_size-write_offset);
	if (log_level > 3) {
	    sprintf(s, "W: writing block %d. Write offset %d.", tr, write_offset);
	    write_log(s);
	}
	c = write(dv, write_buffer, block_size);/* write data */
	blocks_passed++;
	tr += (block_size - write_offset);
	if (c == -1) {				 /* error */
	    if (errno == ENOSPC)		 /* no space on device */
	      c = 0;			 /* means no bytes written */
	    else {				 /* someo ther error */
		sprintf(s, "W: Error %d doing writing. dv = %d", errno, dv);
		if (log_level > 3) write_log(s);
		return -1;
	    }
	}
	
	if (c == block_size) {		 /* wrote whole block successfully */
	    write_offset = 0;		 /* nothing left in write buffer */
	    continue;
	}						
	for (c1=0; c1<c; c1++)		 /* move data that wasn't written to  */
	  write_buffer[c1] = write_buffer[c1+c];/* beginning of buffer */
	write_offset = block_size - c;
	left_tr = tr;				 /* tell parent how much is left */
	return -2;				 /* tell parent couldn't write it all*/
    }
    left_tr = 0;
    if (log_level > 3) write_log("W: finished writing & exiting");
    return 0;
}


_errstat do_next_tape()
{
/* The child ran out of space while trying to write. Must
 * be end of tape. Get new tape and write out what's left
 * in the buffer. Returns -1 if error, 0 otherwise
 */
    WINDOW *mes=NULL;
    char *buf_to_write, s[MAXNAMLEN];
    struct tape_header tdh1;
    _s32 c;
    
    if (log_level > 2) write_log("End of tape reached while writing.");
    tape_close_engine(0);
    buf_to_write = (w_current_buffer == w_buffer_1) ?
      w_buffer_2 : w_buffer_1;
    mes = status_box(mes, "Identifying tape", 1, TRUE, 1);
    while (1) {
	message_box("Insert next tape", MB_OK); paint_main();
	if (tape_rewind() == -1) return -1;
	if (tape_open_engine(O_RDWR, 0) == -1) {
	    write_error_log("Error opening new tape");
	    continue;
	}
	tdh1.magic = 0;
	tape_readheader(&tdh1, 1);
	if (tdh1.magic == TAPER_MAGIC_NUMBER) {
	    if (tdh1.archive_id == 0) 
	      break;
	    else
	      if (message_box("This is tape contains taper data. Confirm overwrite", MB_YESNO))
	        break;
	}
	else {					 /* freshly formatted taper tape */
	    if (message_box("Unknown tape data. Confirm overwrite", MB_YESNO))
	      break;
	}
    }
    
    mes = status_box(mes, "Erasing tape", 1, FALSE, 1);   
    tape_erase();				 /* erase tape */
    status_box(mes, "Rewinding tape & writing tape header", 1, FALSE, 1);   
    tape_close_engine(0);
	    
    if (log_level > 2) write_log("Opening new tape for data writing");
    if (tape_open_engine(O_RDWR, 0) == -1) {
	write_error_log("Error opening new tape");
	close_statusbox(mes);
	return -1;
    }
    
    tdh.tape_number++;
    ifd.number_tapes++;
    blocks_passed = 0;				 /* reset blocks passed */
    if (write_tape_header(&tdh) == -1)  {
	close_statusbox(mes);
	return -1;
    }

    status_box(mes, "Writing data to new tape", 1, FALSE, 1);
    sprintf(s, "Writing out remaining %d bytes to new tape", buf_length-left_tr);
    if (log_level > 2) write_log(s);
    c= do_write_block(buf_to_write+left_tr, buf_length-left_tr,
				 (w_cur_pos < DOUBLE_BUFFER_SIZE) ?
				 1 : 0);
    close_statusbox(mes);
    touchwin(win_main); wrefresh(win_main);
    return c;
}



void do_write_child()
{
/* This is the child process that runs in the background. It 
 * sleeps until it write_buf_no becomes non-null. It then
 * assumes that write_buf_no contains the address of the
 * buffer to write, buf_length is how much is in that buffer
 * and for_close tells whether this is done prior to a close
 * operation or not.
 * 
 * The child has to exit if runs out of space since parent
 * needs to be able to close the backup device which it
 * can't do if the child has it open. The child could
 * close it, but then it needs to open the new device
 * and it won't know when/how etc..
 * 
 * If an error occurs, write_buf_no == -1, and the child is
 * killed
 * 
*/
    char s[100];
    _errstat  ret;

    signal(SIGTERM, child_term);
    while (1) {
	if (write_buf_no != NULL) {
	    if (log_level > 3) {
		sprintf(s, "W: Child about to write out buffer %s, length %d, close %c", 
			(write_buf_no == w_buffer_1) ? "buffer 1" : "buffer 2",
			buf_length, '0'+for_close);
		write_log(s);
	    }
	    ret = do_write_block((char *) write_buf_no, buf_length, for_close);
	    if (ret < 0) {
		write_buf_no = (ret == -1) ? (char *) -1 : NULL;   /* -1 if error, NULL if -2 */
		write_pid = 0;			 /* I'm going to exit */
		(ret == -1) ? sprintf(s, "W: Child exiting because of error %d", errno) :
		              sprintf(s, "W: Child exiting because out of space");
		if (log_level > 3) write_log(s);
		child_term();
	    }
	    write_buf_no = NULL;
	}
    }
}


_errstat flush_buffers()
{
/* Flushes the write buffer
 * 
 * An end of tape is assumed if a write returns a count less than 'count'  -- OR --
 * zero bytes written  -- OR --
 * wrote less bytes than we asked for 
 * 
 * Returns -1 for error, 0 otherwise
 * 
 * If called from tape_close, fromclose=1, otherwise=0
 */
#ifdef TRIPLE_BUFFER
    pid_t f;
    volatile char *c1;
#endif
    
    if (!dv) return 0;				 /* backup device not open */
    if (left_tr) 				 /* child ran out of room writing */
	if (do_next_tape() == -1) return -1;	 /* so write out rest of that buffer first */
    if (w_cur_pos < 1) return 0;		 /* nothing in buffer */

#ifdef TRIPLE_BUFFER
    if (write_pid) {				 /* there's a child running */
	c1 = write_buf_no;
	if ((c1 != NULL) && (c1 != (char *) -1)) {	 /* still working on other buffer */
	    if (log_level > 2) write_log("Waiting for buffer to be written out before writing next one");
	    while ((c1 != NULL) && (c1 != (char *) -1)) 
	      c1 = write_buf_no;
	}
    
	if (write_buf_no == (char *) -1) {		 /* a previous write operation */
	    write_log("Write child returned a write error");   /* ended in error */
	    return do_exit(ERROR_WRITING);
	}
    }
    else {
	write_buf_no = NULL;			 /* tell child nothing to start with */
	if (log_level > 3) write_log("About to fork of write child");
	f = fork();
	if (f == 0) {
	    if (log_level > 3) write_log("Write child forked off");
	    do_write_child();			 /* do child loop */
	}
	if (f == -1)  {
	    if (log_level > 1) write_warning_log("Unable to fork - not triple buffering");
	    if (do_write_block(w_current_buffer, w_cur_pos, for_close) == -1)   /* otherwise do it manually */
	      return do_exit(ERROR_WRITING);
	    goto swapbuf;
	}
	write_pid = f;				 /* must be parent */
    }
#endif
    
    if (w_cur_pos < DOUBLE_BUFFER_SIZE) {	 /* pad out rest of buffer */
	memset(w_current_buffer+w_cur_pos+1, 0,  DOUBLE_BUFFER_SIZE-w_cur_pos-1);
	for_close = 1;
    }
    else
      for_close = 0;

#ifdef TRIPLE_BUFFER    
    if (log_level > 2) write_log("Writing out current buffer");
    buf_length = w_cur_pos;
    write_buf_no = w_current_buffer;		 /* this should start off child */
    swapbuf:;
#else						/* if triple buffering */
    if (log_level > 2) write_log("Writing out current buffer");
    if (do_write_block(w_current_buffer, w_cur_pos, for_close) == -1)   /* otherwise do it manually */
      return do_exit(ERROR_WRITING);
    write_buf_no = NULL;
#endif  
    w_current_buffer = (w_current_buffer == w_buffer_1) ? w_buffer_2
      : w_buffer_1;				 /* swap buffers */
    w_cur_pos = 0;
    return 0;
}



int tape_write(_vptr buf, size_t count)
{
    size_t tr_bytes;
    size_t fit_in;
    
    tr_bytes = 0;				 /* either now or previously */
    while (tr_bytes < count) {
	fit_in = min(count-tr_bytes,DOUBLE_BUFFER_SIZE-w_cur_pos);
    	memcpy(w_current_buffer+w_cur_pos, (char *) buf+tr_bytes, fit_in);
	w_cur_pos += fit_in; 
	tr_bytes += fit_in;
	if (w_cur_pos >= DOUBLE_BUFFER_SIZE)
	  if (flush_buffers() == -1)
	    return do_exit(ERROR_WRITING);
    }
    return tr_bytes;
} 


int tape_read(_vptr buf, size_t count)
{
/* An end of tape is assumed if we get an return code=-1 and errno=ENOSPC   --OR--
 * the count returned is less than we asked for 
 * 
 * note that if the return count is 0, then this is assumed to be an
 * end of volume and not and end of tape, so the zero is returned.
*/
    size_t c;
    
    c =  read_from_buf(buf, count);
    if (c == count) return c;			 /* OK - written what we asked for */
    if (c == -1)				 /* error */
      if (errno != ENOSPC)			 /* make sure not an end of tape error */
        return -1;				 /* no - return error */
      else
        c=0;					 /* none would have been written */
    if (c==0) return 0;				 /* assume end of volume */
/* Must be at end of tape */
    if (log_level > 2) write_log("End of tape reached while reading.");
    if (check_tape(NULL, 0, tdh.tape_number+1) == -1) 	 /* get next tape */
      return -1;
    return read_from_buf((char *) buf+c, count-c)+c;	 /* return data */
}


_errstat get_tape_header(WINDOW *mess, int line, struct tape_header *tdh)
{
/* Tries to get the tape header of the tape currently in the drive.
 * 
 * Opens the tape but doesn't close it

   Returns -1 if error
	   BAD_MAGIC   bad magic number
	   TAPE_EXIST  successful return
*/

    memset(tdh, 0, sizeof(*tdh));
    if (mess)
      mess = status_box(mess, "Rewinding tape", line, FALSE, 1);
    if (tape_rewind() == -1) return -1;		 /* rewind tape */
     
    if (mess)
      status_box(mess, "Identifying tape", line, FALSE, 1);
    if (tape_open(O_RDWR) == -1) 		 /* open rewinding device */
      return -1;

    if (tape_readheader(tdh, 1) == -1)   	 /* could open but can't */
	return BAD_MAGIC;			 /* read - therefore is bad magic */
    
    if (tdh->magic != TAPER_MAGIC_NUMBER) {
	if (tdh->magic == TAPER_4_MAGIC_NUMBER)
	  if (mess)
	    message_box("Sorry, taper 4 archives are not compatible", MB_OK);
	return BAD_MAGIC;
    }
    
    if ((tdh->magic == TAPER_MAGIC_NUMBER) && (tdh->archive_id == 0))
	return TAPE_EXIST_EMPTY;
    return TAPE_EXIST;
}


_errstat read_volheader(struct volume_header *vh, _s8 read_vh, _s8 into_mem)
{
/* Reads in volume header. If into_mem, places the information into
 * memory, otherwise discards it. If read_vh, then expects to read
 * the volume header, otherwise assumes that it is already read in.
 * 
 * returns -1 if error, 0 otherwise
*/
    char *z;
    
    if (read_vh) {
	if (log_level > 2) write_log("Reading in volume header");
	if (tape_read((char *) vh, sizeof(struct volume_header)) != sizeof(struct volume_header)) { /* volume magic number */
	    write_fatal_log("reading in volume header");
	    return do_exit(ERROR_READING);
	}
    }
    if (vh->volume_magic != VOLUME_MAGIC)	 /* volume magic doesn't agree - warning */
      write_warning_log("Volume magic number incorrect");   /* try to proceed */
    if (log_level > 2) write_log("Reading in volume header details");
    if (into_mem) {
	vol_headers = my_realloc(vol_headers, ifd.size_volume_headers + vh->size_header);
	if (vol_headers == NULL)
	  return do_exit(ERROR_MEMORY);
	*((struct volume_header *) (vol_headers+ifd.size_volume_headers)) = *vh;
	if (tape_read(vol_headers+ifd.size_volume_headers+sizeof(*vh), vh->size_header-sizeof(*vh)) == -1)
	  return -1;
	return 0;
    }
    z = alloca(vh->size_header);		 /* skip past header details */
    if (z == NULL) return do_exit(ERROR_MEMORY);
    if (tape_read(z, vh->size_header-sizeof(*vh)) == -1) /* read past it */
	return -1;
    return 0;
}


_errstat goto_end_vol(WINDOW *mes, int ln, _s32 vol, _s32 at_vol, _s8 get_th)
{
/* Positions tape at beginning of volume 'vol'+1. Assumes at at_vol
 * Prints status on line 'ln' of box 'mes'
 * If get_th == 1, then:
 *   opens the rewinding device
 *   gets tape header
 * otherwise assumes that's rewinding is open & it's been got
 * 
 * 
 * If !fast_fsf, then the drive is closed & re-opened as
 * non-rewinding & re-positioned, otherwise quickly fsf
 * 
 * Returns -1 if error or abort
 *   otherwise 
 *     if fsf returns the volume # we are at the beginning of
 *     if not fsf returns the number of blocks skipped
*/
    int c, want_tape, pass_blocks, pass_vols;
    struct volume_tape_info *vt;
    char s[100];

    vt = vt_info + vol -1;			 /* get information of  */
    want_tape = vt->end_tape;
    pass_blocks = 0;
    pass_vols = 0;
    vt = vt_info;
    for (c=0;c<vol;c++) {			 /* work out how many blocks to pass */
	if ((vt->end_tape == want_tape) && (vt->volume <= vol)) {
	    pass_blocks += vt->blocks_on_last_tape;
	    pass_vols++;
	}
	vt++;
    }

    if (get_th) 				 /* only get if user wants */
      while (1) {				 /* use to get tape header */
	  c = get_tape_header(mes, ln, &tdh);
	  at_vol = 1;
	  if (c==-1) return -1;
	  if (c==BAD_MAGIC) {
	      message_box("This is not a taper tape", MB_OK);
	      touchwin(win_main); wrefresh(win_main);
	  }
	  if (c==TAPE_EXIST) break;
      }

    if (check_tape(mes, ln, want_tape) == -1)	 /* make sure correct tape in drive */
	  return -1;				 /* user aborted or error */
    status_box(mes, "Advancing to correct volume", ln, FALSE, 1);   
    if (have_fsf) {
	if (!fast_fsf) {
	    tape_close();			 /* close rewinding device opened earlier */
	    if ((dv = ntape_open(O_RDWR)) == -1) /* now open non-rewinding so we can position */
	      return -1;
	    if (tape_fsf(pass_vols) == -1)	 /* reposition */
	      return do_exit(ERROR_TAPE_FSF);
	    tape_close();			 /* close it to effect change */
	    if ((dv = tape_open(O_RDWR)) == -1)	 /* reopen at new position (rewinding device)*/
	      return -1;
	    return pass_vols+1;
	}
	else {					 /* have a fast fsf */
	    if (!(pass_vols-at_vol+1)) return 0;
	    if (tape_fsf(pass_vols-at_vol+1) == -1) 
	      return -1;
	}
	clear_buffer_pointers();
	tape_set_blk_size();		 
	return pass_vols+1;
    }

    if (blocks_passed < pass_blocks) {		 /* ?need to advance */
	for (c=blocks_passed; c<pass_blocks; c++) {   /* yes */
	    sprintf(s, "Passing block %d of %d", c, pass_blocks);
	    status_box(mes, s, ln, FALSE, 1);
	    if (read(dv, read_buffer, block_size) != block_size)
	      return do_exit(ERROR_SKIPPING);
	    blocks_passed++;
	}
    }
    clear_buffer_pointers();			 /* discard whatever was left */
    status_box(mes, "", ln, FALSE, 1);
    return pass_blocks;
}

    
