;**************************************************************************
;*
;* Boot-ROM-Code to load an operating system across a TCP/IP network.
;*
;* Module:  fileio.asm
;* Purpose: File I/O functions for DOS simulator
;* Entries: dos3C, dos3D, dos3E, dos3F, dos40
;*
;**************************************************************************
;*
;* Copyright (C) 1995,1996 Gero Kuhlmann <gero@gkminix.han.de>
;*
;*  This program is free software; you can redistribute it and/or modify
;*  it under the terms of the GNU General Public License as published by
;*  the Free Software Foundation; either version 2 of the License, or
;*  any later version.
;*
;*  This program is distributed in the hope that it will be useful,
;*  but WITHOUT ANY WARRANTY; without even the implied warranty of
;*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;*  GNU General Public License for more details.
;*
;*  You should have received a copy of the GNU General Public License
;*  along with this program; if not, write to the Free Software
;*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
;*


;
;**************************************************************************
;
; Include assembler macros:
;
include ..\..\headers\asm\macros.inc
include ..\..\headers\asm\layout.inc
include ..\..\headers\asm\memory.inc
include .\dospriv.inc


;
;**************************************************************************
;
; Miscelleneous equates
;
CHR_BS		equ	08h		; Backspace character
CHR_CR		equ	0Dh		; Return character
CHR_LF		equ	0Ah		; Line-feed character
CHR_EOF		equ	1Ah		; End-of-file character


;
;**************************************************************************
;
; Data segment
;
data_start

	extrn	curr_psp:word	; segment of current PSP

data_end


;
;**************************************************************************
;
; Start code segment.
;
text_start

	extrn	dos02:near
	extrn	dos03:near
	extrn	dos04:near
	extrn	dos05:near
	extrn	dos07:near

	public	dos3C, dos3D, dos3E	; define entry points
	public	dos3F, dos40


;
;**************************************************************************
;
; Create a new file. Since it's not possible to create a new file with
; the DOS simulator, this just returns an error.
; Input:  CX     -  file attribute
;         DS:DX  -  pointer to file name
; Output: AX     -  file handle or error code if carry flag set
;         Carry flag  -  set if error
; Registers changed: AX
;
dos3C		proc	near

	mov	ax,ERR_NODIR		; return with error
	stc
	ret

dos3C		endp


;
;**************************************************************************
;
; Open file. Since the DOS simulator cannot use files, just return with
; an error.
; Input:  DS:DX  -  pointer to file name
; Output: AX     -  file handle or error code if carry flag set
;         Carry flag  -  set if error
; Registers changed: AX
;
dos3D		proc	near

	mov	ax,ERR_NOFILE		; return with error
	stc
	ret

dos3D		endp


;
;**************************************************************************
;
; Close file. This just clears the file handle in the PSP. If no file IO
; is enabled this is just a NOP.
; Input:  BX  -  file handle
; Output: AX  -  error code if carry flag set
;         Carry flag  -  set if error
; Registers changed: AX
;
dos3E		proc	near

ifdef NEEDFILEIO

	call	chkhndl				; check file handle
	jc	short dos3E9
	push	ds
	mov	ds,curr_psp
	mov	byte ptr ds:psp_handles[bx],0	; reset handle
	pop	ds

else						;; NEEDFILEIO

	clc

endif						;; NEEDFILEIO

dos3E9:	ret

dos3E		endp


;
;**************************************************************************
;
; Read file.
; Input:  BX     -  file handle
;         CX     -  number of bytes to read
;         DS:DX  -  pointer to read buffer
; Output: AX     -  number of bytes read, or error code if carry flag set
;         Carry flag  -  set if error
; Registers changed: AX
;
dos3F		proc	near

ifdef NEEDFILEIO

	call	chkhndl			; check file handle
	jc	short dos3F9

	push	es
	push	di

; Read the data from the device into the buffer.

	mov	di,dx
	mov	es,old_ds[bp]
dos3F3:	call	readchar		; read character from device

; Handle end-of-line or end-of-file.

dos3F4:	cmp	al,CHR_CR		; end of line?
	je	short dos3F5
	cmp	al,CHR_LF
	je	short dos3F5
	cmp	al,CHR_EOF		; or end of file?
	jne	short dos3F6
dos3F5:	mov	al,CHR_CR
	call	writechar		; return to a new line on the screen
	mov	al,CHR_LF
	call	writechar
	mov	ax,di
	sub	ax,dx			; compute actual length of input buffer
	clc
	jmp	short dos3F2

; Handle backspace character

dos3F6:	cmp	al,CHR_BS
	jne	short dos3F7
	cmp	dx,di			; already at beginning of line
	je	short dos3F3
	dec	di
	call	writechar		; print backspace onto the screen
	jmp	short dos3F3

; Add character to the end of the line

dos3F7:	cmp	al,' '			; only care for printable characters
	jb	short dos3F3
	push	ax
	mov	ax,cx
	add	ax,dx			; check if end of buffer reached
	cmp	ax,di
	pop	ax
	jae	short dos3F5
	stosb				; save character into buffer
	call	writechar		; print character
	jmp	short dos3F3

dos3F8:	stc
dos3F2:	pop	di
	pop	es

else

	xor	ax,ax			; if no FILEIO, read just 0 bytes

endif					;; NEEDFILEIO

dos3F9:	ret

dos3F		endp


;
;**************************************************************************
;
; Write to file.
; Input:  BX     -  file handle
;         CX     -  buffer size
;         DS:DX  -  output buffer
; Output: AX     -  number of characters written, or error code if carry set
;         Carry flag  -  set if error
; Registers changed: AX
;
dos40		proc	near

ifdef NEEDFILEIO

	call	chkhndl			; check file handle
	jc	short dos409

	push	es
	push	si

; Write all characters onto the output device

	mov	si,dx
	mov	es,old_ds[bp]
dos402:	mov	ax,dx
	add	ax,cx			; check if at end of buffer
	cmp	si,ax
	jae	short dos403
	lods	byte ptr es:[si]
	call	writechar		; print character
	jmp	short dos402
dos403:	mov	ax,si
	sub	ax,dx			; compute number of bytes written
	clc
	pop	si
	pop	es

else

	xor	ax,ax			; if no FILEIO, just wrote 0 bytes

endif					;; NEEDFILEIO

dos409:	ret

dos40		endp


ifdef NEEDFILEIO
;
;**************************************************************************
;
; Read character from device.
; Input:  BX  -  file handle
; Output: AL  -  character
; Registers changed: AX
;
readchar	proc	near

	mov	ah,07h
ifndef NEEDEXTIO
	jmp	dos07			; read character from keyboard
else
	cmp	bx,RES_STDERR		; check for keyboard input
  ifdef IS386
	jbe	dos07			; read character from keyboard
  else
	ja	short read2
	jmp	dos07			; read character from keyboard
  endif

read2:	cmp	bx,RES_AUX		; check for serial device
  ifdef IS386
	je	dos03			; receive character
  else
	jne	short read9
	jmp	dos03			; receive character
  endif

read9:	ret
endif

readchar	endp
endif					;; NEEDFILEIO


ifdef NEEDFILEIO
;
;**************************************************************************
;
; Write character to device.
; Input:  BX  -  file handle
;         AL  -  character to write
; Output: none
; Registers changed: none
;
writechar	proc	near

	push	dx
	mov	dl,al
ifndef NEEDEXTIO
	call	dos02			; print character
else
	cmp	bx,RES_STDERR		; check for console output
	ja	short write1
	call	dos02			; print character
	jmp	short write9

write1:	cmp	bx,RES_AUX		; check for serial device
	jne	short write2
	call	dos04			; send character
	jmp	short write9

write2:	cmp	bx,RES_PRN		; check for printer device
	jne	short write9
	call	dos05			; send character
endif

write9:	pop	dx
	ret

writechar	endp
endif					;; NEEDFILEIO


ifdef NEEDFILEIO
;
;**************************************************************************
;
; Check if a file handle is correct
; Input:  BX  -  file handle
; Output: AX  -  error code if handle not correct
;         Carry flag  -  set if file handle not correct
; Registers changed: AX
;
chkhndl		proc	near

ifndef NEEDEXTIO
	cmp	bx,RES_AUX		; we can only use console handles
else
	cmp	bx,RES_HANDLES		; we can only use reserved handles
endif
	jae	short chkh8
	push	ds
	mov	ds,curr_psp
	test	byte ptr ds:psp_handles[bx],0FFh
	pop	ds
	jnz	short chkh9		; check if handle is open
chkh8:	mov	ax,ERR_INVHDL
	stc
chkh9:	ret

chkhndl		endp
endif					;; NEEDFILEIO


;
;**************************************************************************
;
text_end

	end

