/* Copyright (c) 1993 by Sanjay Ghemawat */
/*
 * Routines to speed up dateeditor.
 */
#include <stdio.h>
#include <string.h>
#include <tcl.h>

#include "Array.h"
#include "Date.h"
#include "Month.h"
#include "WeekDay.h"

#include "cal_tcl.h"
#include "calendar.h"
#include "calfile.h"
#include "collect.h"
#include "ical.h"
#include "item_tcl.h"

declareArray(HiliteList,char const*)
implementArray(HiliteList,char const*)

static int monday_first();
static int contains(HiliteList const&, char const*);

/*
 * requires	argc/argv === <cmd> <canvas> <date>
 *		Canvas has 42 text items with tags ranging from
 *		=1 to =42.  The text items are arranged in a grid
 *		7 wide by 6 high.  The numbers increase from left to
 *		right and then top to bottom.  Therefore, the top-left
 *		item has tag =1, rop-right item has tag =7, and the
 *		bottom-right item tag =42.  Each one of these 42 items
 *		also has the tag "Day".
 *
 * effects	Configures items so that their text contents correspond
 *		to the day of the month that should be displayed at that
 *		item (under the assumption that the month  named by <date>
 *		is being displayed).
 */
int Cmd_MonthDays(ClientData, Tcl_Interp* tcl, int argc, char* argv[]) {
    if (argc != 3) {
	TCL_Error(tcl, "illegal number of arguments");
    }

    int dateDays;
    if (Tcl_GetInt(tcl, argv[2], &dateDays) != TCL_OK) {
	return TCL_ERROR;
    }
    Date date(dateDays);
    Date first(1, date.GetMonth(), date.GetYear());

    int start = first.GetWDay().Index();
    if (monday_first()) {
	start = ((start + 5)%7) + 1;
    }
    int finish = start + first.GetMonth().Size(first.GetYear()) - 1;

    char* canvas = argv[1];
    if (Tcl_VarEval(tcl, canvas, " itemconfig Day -text {}", NULL) != TCL_OK) {
	return TCL_ERROR;
    }

    for (int i = start; i <= finish; i++) {
	char str[100];
	sprintf(str, "=%d -text {%4d}", i, i - start + 1);
	if (Tcl_VarEval(tcl, canvas, " itemconfig ", str, NULL) != TCL_OK) {
	    return TCL_ERROR;
	}
    }

    TCL_Return(tcl, "");
}

/*
 * requires	"argv == <cmd> <hlist> <start> <finish> <dvar> <hvar> <body>"
 * effects	For each item occurrence in "<start>..<finish>",
 *		set "<dvar>" to item date, "<hvar>" to hilite string for
 *		item, and execute "<body>".  If the item hilite is not in
 *		"<hlist>", the string "always" is used instead of actual
 *		hilite style.
 */
int Cmd_HiliteLoop(ClientData, Tcl_Interp* tcl, int argc, char* argv[]) {
    int i;

    if (argc != 7) {
	TCL_Error(tcl, "illegal number of arguments");
    }

    int startDays;
    if (Tcl_GetInt(tcl, argv[2], &startDays) != TCL_OK) {
	return TCL_ERROR;
    }
    int finishDays;
    if (Tcl_GetInt(tcl, argv[3], &finishDays) != TCL_OK) {
	return TCL_ERROR;
    }

    int count;
    char** strlist;
    if (Tcl_SplitList(tcl, argv[1], &count, &strlist) != TCL_OK) {
	return TCL_ERROR;
    }
    HiliteList hlist;
    for (i = 0; i < count; i++)
	hlist.append(strlist[i]);

    Date today = Date::Today();
    Date start(startDays);
    Date finish(finishDays);
    char* dvar = argv[4];
    char* hvar = argv[5];
    char* body = argv[6];

    Occurrences list;
    collect_occurrences(list, start, finish, 0);

    for (i = 0; i < list.size(); i++) {
	char const* hilite = list[i].item->value()->Hilite();
	if (strcmp(hilite, "never") == 0) continue;
	if (strcmp(hilite, "expire") == 0) {
	    if (list[i].date < today) continue;
	    hilite = "always";
	}
	if (!contains(hlist, hilite)) hilite = "always";

	char buffer[20];
	sprintf(buffer, "%d", list[i].date.EpochDays());
	if (Tcl_SetVar(tcl, dvar, buffer, 0) == NULL) {
	    free((char*) strlist);
	    TCL_Error(tcl, "could not set loop variable");
	}

	if (Tcl_SetVar(tcl, hvar, (char*)hilite, 0) == NULL) {
	    free((char*) strlist);
	    TCL_Error(tcl, "could not set loop variable");
	}


	int result = Tcl_Eval(tcl, body);

	if (result == TCL_OK) continue;
	if (result == TCL_CONTINUE) continue;
	if (result == TCL_BREAK) break;

	// Error of some sort
	free((char*) strlist);
	return result;
    }

    free((char*) strlist);
    TCL_Return(tcl, "");
}

/*
 * requires	argc/argv === <cmd> <canvas> <date>
 *		Canvas has 42 text items with tags ranging from
 *		=1 to =42.  The text items are arranged in a grid
 *		7 wide by 6 high.  The numbers increase from left to
 *		right and then top to bottom.  Therefore, the top-left
 *		item has tag =1, rop-right item has tag =7, and the
 *		bottom-right item tag =42.  Each one of these 42 items
 *		also has the tag "Day".
 *
 * effects	Configures items so that they have tag "interest" iff
 *		the corresponding date has interesting items.  Also
 *		configure the items to have tag "holiday" iff the
 *		corresponding date has an item marked as a holiday item.
 */
int Cmd_InterestTags(ClientData, Tcl_Interp* tcl, int argc, char* argv[]) {
    if (argc != 3) {
	TCL_Error(tcl, "illegal number of arguments");
    }

    int dateDays;
    if (Tcl_GetInt(tcl, argv[2], &dateDays) != TCL_OK) {
	return TCL_ERROR;
    }
    Date date(dateDays);
    Date first(1, date.GetMonth(), date.GetYear());
    int size = first.GetMonth().Size(first.GetYear());
    Date last = first + size - 1;

    /* Number of initial blank entries in display */
    int blanks = first.GetWDay().Index() - 1;
    if (monday_first()) {
	blanks = (blanks + 6)%7;
    }

    int interest[43], holiday[43];
    for (int i = 1; i <= 42; i++) {
	interest[i] = 0;
	holiday[i] = 0;
    }

    Date today = Date::Today();
    Occurrences list;
    collect_occurrences(list, first, last, 0);
    for (i = 0; i < list.size(); i++) {
	int index = (list[i].date - first) + blanks + 1;

	char const* hilite = list[i].item->value()->Hilite();

	if (strcmp(hilite, "never") == 0) {
	    continue;
	}

	if (strcmp(hilite, "holiday") == 0) {
	    holiday[index] = 1;
	    continue;
	}

	if (strcmp(hilite, "expire") == 0) {
	    if (list[i].date >= today) {
		interest[index] = 1;
	    }
	    continue;
	}

	interest[index] = 1;
    }

    char* canvas = argv[1];

    /* Add interest to everything that we are still interested in */
    for (i = 1; i <= 42; i++) {
	char str[100];
	sprintf(str, "=%d", i);

	if (interest[i]) {
	    if (Tcl_VarEval(tcl, canvas, " addtag interest withtag ", str,
			    NULL) != TCL_OK) {
		return TCL_ERROR;
	    }
	}

	if (holiday[i]) {
	    if (Tcl_VarEval(tcl, canvas, " addtag holiday withtag ", str,
			    NULL) != TCL_OK) {
		return TCL_ERROR;
	    }
	}
    }

    TCL_Return(tcl, "");
}

static int monday_first() {
    Calendar* cal = calendar_instance->main->GetCalendar();
    return ((strcmp(cal->GetOption("MondayFirst"), "1") == 0) ? 1 : 0);
}

int contains(HiliteList const& list, char const* hilite) {
    for (int i = 0; i < list.size(); i++) {
	if (strcmp(list[i], hilite) == 0) return 1;
    }
    return 0;
}
