/*
 * Copyright (c) 1980 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/* Modified for the Linux SCSI tape driver by Kai Makisara.
   Last Modified: Sun Feb 19 21:32:04 1995 by root@kai.home
*/

#ifndef lint
char copyright[] =
"@(#) Copyright (c) 1980 The Regents of the University of California.\n\
 All rights reserved.\n";
#endif /* not lint */

#ifndef lint
static char sccsid[] = "@(#)mt.c	5.6 (Berkeley) 6/6/91";
#endif /* not lint */

/*
 * mt --
 *   magnetic tape manipulation program
 */
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/mtio.h>
#include <fcntl.h>
#include <stdio.h>
#include <ctype.h>

#define	equal(s1,s2)	(strcmp(s1, s2) == 0)

void status();
void prt_densities();

struct commands {
	char *c_name;
	int c_code;
	int c_count_bits;
	int c_ronly;
} com[] = {
	{ "weof",	MTWEOF,	0, 0 },
	{ "wset",	MTWSM,	0, 0 },
	{ "eof",	MTWEOF,	0, 0 },
	{ "fsf",	MTFSF,	0, 1 },
	{ "fsfm",	MTFSFM,	0, 1 },
	{ "bsf",	MTBSF,	0, 1 },
	{ "bsfm",	MTBSFM,	0, 1 },
	{ "fsr",	MTFSR,	0, 1 },
	{ "bsr",	MTBSR,	0, 1 },
	{ "fss",	MTFSS,	0, 1 },
	{ "bss",	MTBSS,	0, 1 },
	{ "rewind",	MTREW,	0, 1 },
	{ "offline",	MTOFFL,	0, 1 },
	{ "rewoffl",	MTOFFL,	0, 1 },
	{ "retension",	MTRETEN,0, 1 },
	{ "eod",	MTEOM,	0, 1 },
	{ "seod",	MTEOM,	0, 1 },
	{ "seek",	MTSEEK, 0, 1 },
	{ "tell",	MTTELL,	0, 1 },
	{ "status",	MTNOP,	0, 1 },
	{ "erase",	MTERASE, 0, 0},
	{ "setblk",	MTSETBLK, 0, 1},
	{ "setdensity",	MTSETDENSITY, 0, 1},
	{ "drvbuffer",	MTSETDRVBUFFER, 0, 1},
	{ "stoptions",	MTSETDRVBUFFER, MT_ST_BOOLEANS, 0},
	{ "stwrthreshold", MTSETDRVBUFFER, MT_ST_WRITE_THRESHOLD, 0},
	{ 0 }
};

struct densities {
	int code;
	char *name;
} density_tbl[] = {
	{0x00, "default"},
	{0x01, "NRZI (800 bpi)"},
	{0x02, "PE (1600 bpi)"},
	{0x03, "GCR (6250 bpi)"},
	{0x05, "QIC-45/60 (GCR, 8000 bpi)"},
	{0x06, "PE (3200 bpi)"},
	{0x07, "IMFM (6400 bpi)"},
	{0x08, "GCR (8000 bpi)"},
	{0x09, "GCR /37871 bpi)"},
	{0x0a, "MFM (6667 bpi)"},
	{0x0b, "PE (1600 bpi)"},
	{0x0c, "GCR (12960 bpi)"},
	{0x0d, "GCR (25380 bpi)"},
	{0x0f, "QIC-120 (GCR 10000 bpi)"},
	{0x10, "QIC-150/250 (GCR 10000 bpi)"},
	{0x11, "QIC-320/525 (GCR 16000 bpi)"},
	{0x12, "QIC-1350 (RLL 51667 bpi)"},
	{0x13, "DDS (61000 bpi)"},
	{0x14, "EXB-8200 (RLL 43245 bpi)"},
	{0x15, "EXB-8500 (RLL 45434 bpi)"},
	{0x16, "MFM 10000 bpi"},
	{0x17, "MFM 42500 bpi"},
	{0x24, "DDS-2"},
	{140, "EXB-8505 compressed"},
	{144, "EXB-8205 compressed"},
	{-1, NULL}};


void usage(int explain)
{
  fprintf(stderr, "usage: mt [ -f device ] command [ count ]\n");
  if (explain) {
    fprintf(stderr, "commands: weof, eof, wset, fsf, fsfm, bsf, bsfm, fsr, bsr,\n");
    fprintf(stderr, "          fss, bss, rew[ind], offl[ine], rewoffl, reten[sion],\n");
    fprintf(stderr, "          eof, seod, seek, tell, stat[us], erase, setblk,\n");
    fprintf(stderr, "          setdensity, drvbuffer, stoptions, stwrthreshold,\n");
    fprintf(stderr, "          densities\n");
  }
}



int mtfd;
struct mtop mt_com;
struct mtget mt_status;
struct mtpos mt_pos;
char *tape;

main(int argc, char **argv)
{
	int c_code, c_ronly;
	char line[80], *getenv();
	register char *cp;
	register struct commands *comp;

	if (argc > 2 && (equal(argv[1], "-t") || equal(argv[1], "-f"))) {
		argc -= 2;
		tape = argv[2];
		argv += 2;
	} else
		if ((tape = getenv("TAPE")) == NULL)
			tape = DEFTAPE;
	if (argc < 2) {
		usage(0);
		exit(1);
	}
	cp = argv[1];
	if (!strncmp(cp, "datcompression", 4)) {
	  c_code = 1000;
	  c_ronly = 0;
	}
	else if (!strncmp(cp, "densities", 9)) {
	  prt_densities();
	  exit(0);
	}
	else {
	  for (comp = com; comp->c_name != NULL; comp++)
	    if (strncmp(cp, comp->c_name, strlen(cp)) == 0)
	      break;
	  if (comp->c_name == NULL) {
	    fprintf(stderr, "mt: unknown command \"%s\"\n", cp);
	    usage(1);
	    exit(1);
	  }
	  c_code = comp->c_code;
	  c_ronly = comp->c_ronly;
	}
	if ((mtfd = open(tape, c_ronly ? O_RDONLY : O_RDWR)) < 0) {
		perror(tape);
		exit(1);
	}
	if (c_code == 1000) {
	  if (!do_dat_compression(mtfd, (argc > 2), argv[2]))
	    exit(2);
	}
	else if (c_code == MTTELL) {
		if (ioctl(mtfd, MTIOCPOS, (char *)&mt_pos) < 0) {
			fprintf(stderr, "%s %s %d ", tape, comp->c_name,
				mt_pos.mt_blkno);
			perror("failed");
			exit(2);
		}
		printf("At block %d.\n", mt_pos.mt_blkno);
	}
	else if (c_code != MTNOP) {
		mt_com.mt_op = comp->c_code;
		mt_com.mt_count = (argc > 2 ? atoi(argv[2]) : 1);
		mt_com.mt_count |= comp->c_count_bits;
		if (mt_com.mt_count < 0) {
			fprintf(stderr, "mt: negative repeat count\n");
			exit(1);
		}
		if (ioctl(mtfd, MTIOCTOP, (char *)&mt_com) < 0) {
			fprintf(stderr, "%s %s %d ", tape, comp->c_name,
				mt_com.mt_count);
			perror("failed");
			exit(2);
		}
	} else {
		if (ioctl(mtfd, MTIOCGET, (char *)&mt_status) < 0) {
			perror("mt");
			exit(2);
		}
		status(&mt_status);
	}
	return 0;
}

#ifdef vax
#include <vaxmba/mtreg.h>
#include <vaxmba/htreg.h>

#include <vaxuba/utreg.h>
#include <vaxuba/tmreg.h>
#undef b_repcnt		/* argh */
#include <vaxuba/tsreg.h>
#endif

#ifdef sun
#include <sundev/tmreg.h>
#include <sundev/arreg.h>
#endif

#ifdef tahoe
#include <tahoe/vba/cyreg.h>
#endif

struct tape_desc {
	short	t_type;		/* type of magtape device */
	char	*t_name;	/* printing name */
	char	*t_dsbits;	/* "drive status" register */
	char	*t_erbits;	/* "error" register */
} tapes[] = {
#ifdef vax
	{ MT_ISTS,	"ts11",		0,		TSXS0_BITS },
	{ MT_ISHT,	"tm03",		HTDS_BITS,	HTER_BITS },
	{ MT_ISTM,	"tm11",		0,		TMER_BITS },
	{ MT_ISMT,	"tu78",		MTDS_BITS,	0 },
	{ MT_ISUT,	"tu45",		UTDS_BITS,	UTER_BITS },
#endif
#ifdef sun
	{ MT_ISCPC,	"TapeMaster",	TMS_BITS,	0 },
	{ MT_ISAR,	"Archive",	ARCH_CTRL_BITS,	ARCH_BITS },
#endif
#ifdef tahoe
	{ MT_ISCY,	"cipher",	CYS_BITS,	CYCW_BITS },
#endif
#ifdef linux
	{ MT_ISSCSI1,	"SCSI 1",	0,		0 },
	{ MT_ISSCSI2,   "SCSI 2",	0,		0 },
#endif
	{ 0 }
};

/*
 * Interpret the status buffer returned
 */
void status(struct mtget *bp)
{
	struct tape_desc *mt;
	int dens, i;
	char buf[32], *density;

	for (mt = tapes; mt->t_type; mt++)
		if (mt->t_type == bp->mt_type)
			break;
	if (mt->t_type == 0) {
		printf("unknown tape drive type (%d)\n", bp->mt_type);
		return;
	}
	printf("%s tape drive, residual=%d\n", mt->t_name, bp->mt_resid);
#ifdef linux
	printf("File number=%d, block number=%d.\n", bp->mt_fileno, bp->mt_blkno);
	if (bp->mt_type == MT_ISSCSI1 ||
	    bp->mt_type == MT_ISSCSI2) {
	  dens = (bp->mt_dsreg & MT_ST_DENSITY_MASK) >> MT_ST_DENSITY_SHIFT;
	  density = "unknown";
	  for (i=0; density_tbl[i].code >= 0; i++)
	    if (density_tbl[i].code == dens) {
	      density = density_tbl[i].name;
	      break;
	    }
	  printf("Tape block size %d bytes. Density code 0x%x (%s).\n",
		 ((bp->mt_dsreg & MT_ST_BLKSIZE_MASK) >> MT_ST_BLKSIZE_SHIFT),
		 dens, density);

	  printf("Soft error count since last status=%d\n",
		 (bp->mt_erreg & MT_ST_SOFTERR_MASK) >> MT_ST_SOFTERR_SHIFT);
	  printf("General status bits on (%x):\n", bp->mt_gstat);
	  if (GMT_EOF(bp->mt_gstat))
	    printf(" EOF");
	  if (GMT_BOT(bp->mt_gstat))
	    printf(" BOT");
	  if (GMT_EOT(bp->mt_gstat))
	    printf(" EOT");
	  if (GMT_SM(bp->mt_gstat))
	    printf(" SM");
	  if (GMT_EOD(bp->mt_gstat))
	    printf(" EOD");
	  if (GMT_WR_PROT(bp->mt_gstat))
	    printf(" WR_PROT");
	  if (GMT_ONLINE(bp->mt_gstat))
	    printf(" ONLINE");
	  if (GMT_D_6250(bp->mt_gstat))
	    printf(" D_6250");
	  if (GMT_D_1600(bp->mt_gstat))
	    printf(" D_1600");
	  if (GMT_D_800(bp->mt_gstat))
	    printf(" D_800");
	  if (GMT_DR_OPEN(bp->mt_gstat))
	    printf(" DR_OPEN");	  
	  if (GMT_IM_REP_EN(bp->mt_gstat))
	    printf(" IM_REP_EN");
	}
	else {
	  printf("ds=   %0x\n", bp->mt_dsreg);
	  printf("gstat=%0x\n", bp->mt_gstat);
	  printf("ner=  %0x\n", bp->mt_erreg);
	}
#else
	printreg("ds", bp->mt_dsreg, mt->t_dsbits);
	printreg("\ner", bp->mt_erreg, mt->t_erbits);
#endif
	putchar('\n');
}

/*
 * Print a register a la the %b format of the kernel's printf
 */
printreg(s, v, bits)
	char *s;
	register char *bits;
	register unsigned short v;
{
	register int i, any = 0;
	register char c;

	if (bits && *bits == 8)
		printf("%s=%o", s, v);
	else
		printf("%s=%x", s, v);
	bits++;
	if (v && bits) {
		putchar('<');
		while (i = *bits++) {
			if (v & (1 << (i-1))) {
				if (any)
					putchar(',');
				any = 1;
				for (; (c = *bits) > 32; bits++)
					putchar(c);
			} else
				for (; *bits > 32; bits++)
					;
		}
		putchar('>');
	}
}

/*** Get and set the DAT compression (Mode Page 15) ***/

#define MODE_SENSE 0x1a
#define MODE_SELECT 0x15

int read_mode_page(int fn, int page, int length, unsigned char *buffer,
		   int do_mask)
{
  int result, *ip;
  unsigned char tmpbuffer[30], *cmd;

  memset(tmpbuffer, 0, 14);
  ip = (int *)&(tmpbuffer[0]);
  *ip = 0;
  *(ip+1) = length + 4;

  cmd = &(tmpbuffer[8]);
  cmd[0] = MODE_SENSE;
  cmd[1] = 8;
  cmd[2] = page;
  if (do_mask)
    cmd[2] |= 0x40;  /* Get changeable parameter mask */
  cmd[4] = length + 4;

  result = ioctl(fn, 1, tmpbuffer);
  if (result) {
    fprintf(stderr, "Can't read mode page. Are you sure you are root?\n");
    return 0;
  }
  memcpy(buffer, tmpbuffer + 8, length + 4);
  return 1;
}


int write_mode_page(int fn, int page, int length, unsigned char *buffer)
{
  int result, *ip;
  unsigned char tmpbuffer[40], *cmd;

  memset(tmpbuffer, 0, 14);
  ip = (int *)&(tmpbuffer[0]);
  *ip = length + 4;
  *(ip+1) = 0;

  cmd = &(tmpbuffer[8]);
  cmd[0] = MODE_SELECT;
  cmd[1] = 0x10;
  cmd[4] = length + 4;

  memcpy(tmpbuffer + 14, buffer, length + 4);
  tmpbuffer[14] = 0;  /* reserved data length */
  tmpbuffer[18] &= 0x3f;  /* reserved bits in page code byte */

  result = ioctl(fn, 1, tmpbuffer);
  if (result) {
    fprintf(stderr, "Can't write mode page.\n");
    return 0;
  }
  return 1;
}


int do_dat_compression(int fn, int alter, char *new)
{
  int i;
  unsigned char buffer[30], mask[30];

  if (!read_mode_page(fn, 0x0f, 16, buffer, 0)) {
    close(fn);
    return 0;
  }

  if (alter) {
    if (*new == '0')
      buffer[4+2] &= 0x7f;
    else
      buffer[4+2] |= 0x80;
    if (read_mode_page(fn, 0x0f, 16, mask, 1))
      for (i=2; i < 16; i++)
	buffer[4+i] != mask[4+i];
    if (!write_mode_page(fn, 0x0f, 16, buffer)) {
      close(fn);
      return 0;
    }
    if (!read_mode_page(fn, 0x0f, 16, buffer, 0)) {  /* Re-read to check */
      close(fn);
      return 0;
    }
  }

  if (buffer[4+2] & 0x80)
    printf("Compression on.\n");
  else
    printf("Compression off.\n");

  close(fn);
  return 1;
}


void prt_densities()
{
  int i;

  printf("Some SCSI tape density codes:\ncode   explanation\n");
  for (i=0; density_tbl[i].code >= 0; i++)
    printf("0x%02x   %s\n", density_tbl[i].code, density_tbl[i].name);
}
