/* last.c */

/* Copyright (C) 1993 Free Software Foundation, Inc.

This file is part of the GNU Accounting Utilities

The GNU Accounting Utilities are free software; you can redistribute
them and/or modify them under the terms of the GNU General Public
License as published by the Free Software Foundation; either version
2, or (at your option) any later version.

The GNU Accounting Utilities are distributed in the hope that they will
be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with the GNU Accounting Utilities; see the file COPYING.  If
not, write to the Free Software Foundation, 675 Mass Ave, Cambridge,
MA 02139, USA.  */

#include "config.h"
#include <stdio.h>

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#include <string.h>
#include <sys/types.h>
#include <utmp.h>
#include <time.h>

#ifdef HAVE_LIMITS_H
#include <limits.h>
#endif

#ifdef AMIX
#include </usr/ucbinclude/sys/signal.h>
#else
#include <signal.h>
#endif

#include "common.h"
#include "getopt.h"

/* ye olde defines */

#define STILL_ON 0
#define SHUTDOWN 1
#define REBOOT 2
#define REGULAR 3
#define SPECIAL 4
#define SUPPLANT 5

#define BUFFERED_RECS 256

/* define some data structures */

struct hash_entry {
  struct hash_entry *next;
  char ut_line[8];
  long time;
  int fake_entry;
};

#if !defined (SVR4) && !defined (_POSIX_SOURCE)
RETSIGTYPE handler (int, int, struct sigcontext *);
#else
RETSIGTYPE handler (int);
#endif


/* globals */

static char rcsid[] = "$Id: last.c,v 1.9 1994/05/16 10:03:31 noel Exp noel $";

int debugging_enabled;		/* print internal information relating
				 * to entries read, data structures,
				 * etc. */

long time_warp_leniency;	/* number of seconds that an entry can
				 * be "before" the previous one -- to
				 * deal with simultaneous init(8)
				 * writes, or to overlook problems
				 * with sun's wtmp files.  Defaults to
				 * 60. */

int print_file_inconsistencies;	/* when there's a problem with the
				 * file's format, scream to stderr! */

int print_supplant;		/* if a tty entry will be supplanted,
				 * there isn't a username which to
				 * attribute the use of the machine.
				 * Specifying this option makes last()
				 * print these records. */

int last_event;			/* set to STILL_ON initially, but to
				 * REBOOT or SHUTDOWN as they come
				 * along */

struct hash_entry last_down;	/* last time the machine rebooted/shutdown */

long num_lines_to_print;	/* the -n option -- starts out at -1,
				 * which is a signal to the program
				 * not to limit the number of lines
				 * printed. */

char *program_name;		/* name of the program, for usage */

long last_time;			/* time of last event (error checking)
				 * needs to be a global so that the
				 * signal handler can get at its value
				 * when the program is interrupted */

int truncate_ftp_entries;	/* most last progs do this by default,
				 * but we should have a flag for it */

#include "files.h"

/* system V machines have a specific variable in their utmp structure
 * that tells us what type of record it is.  bsd machines don't... */

#ifdef SYSTEM_V_STYLE_RECORDS

#define RECORD_TYPE ut_type
int print_all_records;		/* flag if we want to print things
				 * other than user records */

#else

#define RECORD_TYPE ut_line[0]
#define OLD_TIME '|'
#define NEW_TIME '{'
#define BOOT_TIME '~'

#endif

struct name_entry *name_list;	/* storage for ttys and names */
struct hash_entry *hash_table;

FILE *Wfile;                     /* pointer to the wtmp file */
char default_wtmp_file[] = WTMP_FILE;  /* self-explanatory */
char *wtmp_file_name = default_wtmp_file;    /* the file name */
long line_number = 0;            /* the line number of the file, for
				  * error reporting purposes */
struct utmp *rec_buffer;         /* buffer for read records so that we
				  * don't have to slow things down
				  * with a million fseek calls! */
int buf_last;                    /* number of last item in the record
				  * buffer */

/* prototypes */

void log_everyone_in (long, int, char *);
void give_usage (void);
void parse_entries (void);
void update_system_time (long);
void init_flags_and_data (void);
void log_out (struct utmp *, int);
void log_in (struct utmp *);
void print_record (struct utmp *, struct hash_entry *, int);
int get_entry (struct utmp **);



/* code */

void
main (int argc, char *argv[])
{
  int c;

  init_flags_and_data ();
  program_name = argv[0];
  
  while (1)
    {
      int option_index = 0;
      
      static struct option long_options[] = {
	{ "complain", no_argument, NULL, 1 },
	{ "supplants", no_argument, NULL, 2 },
	{ "lines", required_argument, NULL, 3 },
	{ "debug", no_argument, NULL, 4 },
	{ "timewarp-value", required_argument, NULL, 5 },
	{ "version", no_argument, NULL, 6 },
	{ "help", no_argument, NULL, 7 },
	{ "other-file", required_argument, NULL, 8 },
	{ "no-truncate-ftp-entries", no_argument, NULL, 9 },
#ifdef SYSTEM_V_STYLE_RECORDS
	{ "all-records", no_argument, NULL, 10 },
#endif
	{ 0, 0, 0, 0 }
      };
      
#ifdef SYSTEM_V_STYLE_RECORDS
      c = getopt_long (argc, argv, "an:w:", long_options, &option_index);
#else
      c = getopt_long (argc, argv, "n:w:", long_options, &option_index);
#endif
      
      if (c == EOF)
	break;

      switch (c)
	{
	case 1:
	  print_file_inconsistencies = 1;
	  break;
	case 2:
	  print_supplant = 1;
	  break;
	case 'n':
	case 3:
	  num_lines_to_print = atol (optarg);
	  if (num_lines_to_print < 1)
	    fatal ("number of lines to print must be positive and non-zero");
	  break;
	case 4:
	  debugging_enabled = 1;
	  print_file_inconsistencies = 1;
	  break;
	case 5:
	  time_warp_leniency = atol (optarg);
	  if (time_warp_leniency < 0)
	    fatal ("time warp leniency value has to be non-negative");
	  break;
	case 6:
	  fprintf (stderr, "%s\n", rcsid);
	  break;
	case 7:
	  give_usage ();
	  exit (1);
	  break;
	case 'w':
	case 8:
	  wtmp_file_name = optarg;
	  break;
	case 9:
	  truncate_ftp_entries = 0;
	  break;
#ifdef SYSTEM_V_STYLE_RECORDS
	case 'a':
	case 10:
	  print_all_records = 1;
	  break;
#endif
	default:
	  {
	    /* char ts[255];
	    sprintf (ts, "?? getopt returned char code 0%o ??", c);
	    fatal (ts); */
	    give_usage ();
	    exit (1);
	  }
	  break;
	}
    }

  while (optind < argc)
    {
      /* if we get here, we expect everything else to be a username or tty */
      struct name_entry *node, *node2;
      char *string;
      
      /* make some space! */
      node = (struct name_entry *) xmalloc (sizeof (struct name_entry));
      node2 = (struct name_entry *) xmalloc (sizeof (struct name_entry));
      string = (char *) xmalloc ((strlen(*argv) + 4) * sizeof (char));
      
      /* add the node as-is */
      node->name = argv[optind];
      node->next = name_list;
      name_list = node;
      
      /* add the name modified tty<name> */
      sprintf (string, "tty%s", argv[optind]);
      node2->name = string;
      node2->next = name_list;
      name_list = node2;
      
      optind++;
    }
  
  if (debugging_enabled)
    {
      struct name_entry *np;
      for (np = name_list; np != NULL; np = np->next)
	printf ("%s\n", np->name);
    }
  
  Wfile = open_binary (wtmp_file_name);
  
  (void) signal (SIGINT, (RETSIGTYPE (*)) handler);

  parse_entries ();
  
  fclose (Wfile);

  exit (0);			/* a GUARANTEED return value to the env */
}



/* guess what this does... */
void
give_usage ()
{
  char *usage = "\
Usage: %s [-n <lines>] [-w <file>] [people] [ttys] ...\n\
       [--lines <num>] [--other-file <file>] [--complain] [--supplants]\n\
       [--debug] [--timewarp-value <value>] [--no-truncate-ftp-entries]\n"
#ifdef SYSTEM_V_STYLE_RECORDS
"       [--all-records] [--version] [--help]\n"
#else
"       [--version] [--help]\n"
#endif
;
  
  fprintf (stderr, usage, program_name);
}


/* parse the entries in a wtmp file */
void
parse_entries (void)
{
  struct utmp *rec;           /* the current record */
  int expecting_clock = 0;    /* set to a nonzero value if the next entry
			       * is expected to be the second half of a
			       * clock pair
			       */
  long clock_before = 0;      /* the clock before the date cmd */

  fseek (Wfile, 0, SEEK_END);	/* go to end of file */

#ifndef HAVE_LIMITS_H
  /* set the last time to be one greater than the last entry
     in the wtmp file */

  fseek (Wfile, -sizeof (struct utmp), SEEK_CUR);
  
  if (fread ((void *) rec_buffer, sizeof (struct utmp), 1, Wfile) == -1)
    fatal ("parse_entries: couldn't read from file");
  
  last_time = rec_buffer->ut_time + 1;
  
  fseek (Wfile, 0, SEEK_END);	/* go to end of file */
#endif

  /* loop while there are still entries to read... */
  while (get_entry (&rec))
    {
      if (debugging_enabled)
	{
	  fprintf (stddebug, "---------------------------------------------------------------------------\n");
#ifdef SYSTEM_V_STYLE_RECORDS
	  fprintf (stddebug, "%-8.8s %-8.8s %ld (%d) %s",
		   rec->ut_name, rec->ut_line, (long) rec->ut_time,
		   rec->RECORD_TYPE, ctime ((time_t *) &(rec->ut_time)));
#else
	  fprintf (stddebug, "%-8.8s %-8.8s %ld %s",
		   rec->ut_name, rec->ut_line, (long) rec->ut_time,
		   ctime ((time_t *) &(rec->ut_time)));
#endif
	}
      
      /*
       * if the time of this record occurs AFTER the time of the last
       * record, there's something wrong.  To deal with it, log everyone
       * in at the time of the record.  Unfortunately, this means that
       * statistics cannot be collected for the users on these ttys...
       * Also, reset last_time so THIS check can be made correctly.
       * NOTE: the '|' entry is a special case, so ignore it.
       */
      if ((rec->ut_time - time_warp_leniency > last_time)
	  && (rec->RECORD_TYPE != OLD_TIME))
	{
	  if (print_file_inconsistencies)
	    fprintf (stddebug, "%s:%ld: time warp forwards (%ld/%ld)\n",
		     wtmp_file_name, line_number, last_time, rec->ut_time);
	  
	  log_everyone_in (rec->ut_time, print_file_inconsistencies, "time warp");
	}
      last_time = rec->ut_time; /* so we're ready to run next time */

      /* now, process each record, depeding on its type */
      switch (rec->RECORD_TYPE)
	{
	  
	case OLD_TIME:		/* time after a date command */
	  print_record (rec, NULL, SPECIAL);
	  if (expecting_clock)
	    {
	      update_system_time (rec->ut_time - clock_before);
	      expecting_clock = 0;
	    }
	  else
	    if (print_file_inconsistencies)
	      fprintf (stddebug, "problem: expecting '%c' (clock) record\n",
		       OLD_TIME);
	  break;

	case NEW_TIME:		/* time before a date command */
	  expecting_clock = 1;
	  clock_before = rec->ut_time;
	  print_record (rec, NULL, SPECIAL);
	  break;

	case BOOT_TIME:		/* reboot or shutdown */
	  last_down.time = rec->ut_time;
	  strncpy (last_down.ut_line, rec->ut_line, 8);
	  print_record (rec, NULL, SPECIAL);

	  if (strncmp (rec->ut_name, "shutdown", 8) == 0)
	    {
	      last_event = SHUTDOWN;
	      /* at the given time, the system was shut down
	       * so log everybody in
	       */
	      log_everyone_in (rec->ut_time, debugging_enabled, "shutdown");
	    }
	  else
	    {
	      last_event = REBOOT;
	      log_everyone_in (rec->ut_time, debugging_enabled, "reboot");
	    }
	  break;
	  
#ifdef SYSTEM_V_STYLE_RECORDS

	case USER_PROCESS:
	  log_in (rec);
	  break;

#ifdef DEAD_PROCESS
	case DEAD_PROCESS:
	  log_out (rec, FALSE);
	  break;
#endif

	  /* FIXME: here we assume that any other types are just
             superfluous records (like RUN_LVL, INIT_PROCESS,
             LOGIN_PROCESS, whatever) and they need to be printed out.
             This may not hold true for every arch, however.  So if
             there's a special kind of record that needs to be
             handled, put it (with ifdefs) above.  Also, check the
             PRINT_RECORD function for how to print out special record
             types. */

	default:
	  print_record (rec, NULL, SPECIAL);
	  break;

#else

	default:                 /* login or logout */
	  if (strlen(rec->ut_name) == 0)
	    log_out (rec, FALSE);
	  else
	    log_in (rec);
     
#endif

	}
 
      if (debugging_enabled)
	{
	  /* print out hash table */
	  struct hash_entry *np;
	  for (np = hash_table; np != NULL; np = np->next)
	    {
	      fprintf (stddebug, "**\t%-8.8s %s", np->ut_line,
		       ctime ((time_t *) &(np->time)));
	    }
	}
      
    }

  printf ("\nwtmp begins %-16.16s\n", ctime ((time_t *) &last_time));
}



/* since the sys clock has changed, each entry's login time has to be
 * adjusted...
 */
void
update_system_time (long the_time)
{
  struct hash_entry *np;
  
  if (debugging_enabled)
    fprintf (stddebug, "\t\t\t\t\tupdate by %ld\n", the_time);
  
  for (np = hash_table; np != NULL; np = np->next)
    {
      np->time += the_time;
      
      if (debugging_enabled)
	fprintf (stddebug, "\t\t\t\t\t%ld %ld %-8.8s (clock_chg)\n",
		 np->time, np->time + the_time, np->ut_line);
    }
}


/* log all entries in at THE_TIME.  Update statistics if DEBUG_FLAG
 * is non-zero, and print out the entries preceded by DEBUG_STR.
 */
void
log_everyone_in (long the_time, int debug_flag, char *debug_str)
{
  struct hash_entry *np, *temp;
  
  for (np = hash_table; np != NULL; )
    {
      /* sometimes, when there is a change to/from daylight savings time,
       * the login time is later than the logout time.  KLUDGE: when this
       * happens, add an hour to the logout time
       */
      if (np->time < the_time)
	{
	  if (print_file_inconsistencies)
	    fprintf (stddebug,
		     "problem: login (daylight savings time) %.8s\n",
		     np->ut_line);
	  np->time += 3600;
	}
      
      if (debug_flag)
	fprintf (stddebug, "\t\t\t\t\t\t%-8.8s (%s)\n",
		 np->ut_line, debug_str);

      if (print_supplant)
	{
	  struct utmp this;
	  
	  strncpy (this.ut_line, np->ut_line, 8);
	  this.ut_name[0] = '\0';
#ifdef HAVE_UT_HOST
	  this.ut_host[0] = '\0';
#endif
	  this.ut_time = the_time;
	  
	  print_record (&this, np, SUPPLANT);
      }

      temp = np->next;
      free (np);
      np = temp;
    }
  
  hash_table = NULL;
}


/* put a terminal into the hash table
 * if NO_SUPPLANT_FLAG is true, don't supplant -- just return
 */
void
log_out (struct utmp *entry, int no_supplant_flag)
{
  struct hash_entry *np;

  for (np = hash_table; np != NULL; np = np->next)
    {
      if (strncmp (np->ut_line, entry->ut_line, 8) == 0)
	{
	  if (no_supplant_flag)
	    return;

	  if (!np->fake_entry)
	    {
	      if (print_file_inconsistencies)
		fprintf (stddebug, "problem: duplicate tty record\n");
	  
	      /* since we can't get any information about the username
	       * or hostname of the old entry, just print it out with
	       * question marks in the right places, then supplant the
	       * entry. */

	      if (print_supplant)
		{
		  struct utmp this;
				    
		  strncpy (this.ut_line, entry->ut_line, 8);
		  this.ut_name[0] = '\0';
#ifdef HAVE_UT_HOST
		  this.ut_host[0] = '\0';
#endif
		  this.ut_time = entry->ut_time;
		  
		  print_record (&this, np, SUPPLANT);
		}
	    }

	  strncpy (np->ut_line, entry->ut_line, 8);
	  np->time = entry->ut_time;
	  np->fake_entry = no_supplant_flag;
	  return;
	}
    }
  
  /* if we get here, we didn't find the entry in the list */
  np = (struct hash_entry *) xmalloc (sizeof (struct hash_entry));
  
  /* copy the info */
  strncpy (np->ut_line, entry->ut_line, 8);
  np->time = entry->ut_time;
  np->fake_entry = no_supplant_flag;

  /* insert at the beginning of the list */
  np->next = hash_table;
  hash_table = np;
}


/* log somebody in */
void
log_in (struct utmp *entry)
{
  struct hash_entry *np, *last;
  
  for (np = hash_table, last = NULL; np != NULL; last = np, np = np->next)
    {
      if (strncmp (entry->ut_line, np->ut_line, 8) == 0)
	{
	  /* this is the one! */
	  
	  if (np->time < entry->ut_time)
	    {
	      /* this happens when there is a change to/from daylight
	       * savings time -- result = +/- 1 hour's worth of ac'd
	       * time on those days; we have to kludge here because
	       * it's possible for someone to log out before they've
	       * logged in (according to localtime(3)) -- solution is
	       * to add one hour to the logout time...
	       */
	      np->time += 3600;
	      if (print_file_inconsistencies)
		fprintf (stddebug,
			 "problem: login (daylight savings time) %.8s\n",
			 np->ut_line);
	    }
	  
	  if (np->fake_entry && print_file_inconsistencies)
	    {
	      fprintf (stddebug, "problem: logout time for next entry assumed to be last login time on tty\n");
	    }
	  
	  print_record (entry, np, REGULAR);

	  /* remove the entry from the hash table and make another
	   * logout entry at that time (on that line) if one doesn't
	   * already exist
	   */
	  
	  if (last == NULL)
	    hash_table = np->next;
	  else
	    last->next = np->next;
	  
	  free (np);
	  
	  log_out (entry, TRUE);
	  
	  return;
	}
    }
  /* if we got here, either somebody's still logged in, or there's a
   * problem with the wtmp file
   */
  print_record (entry, &last_down, last_event);

  /* now make another logout entry if one doesn't already exist */
  log_out (entry, TRUE);
}


void
print_record (struct utmp *login, struct hash_entry *logout, int type)
{
  if ((name_list == NULL) || desired_name (login->ut_name, name_list)
      || desired_name (login->ut_line, name_list))
    {
#ifdef SYSTEM_V_STYLE_RECORDS
      if ((login->RECORD_TYPE != USER_PROCESS)
	  && (login->RECORD_TYPE != BOOT_TIME)
	  && !print_all_records)
	return;
#endif

      /* if we're truncating the ftpxxxx entries together, we need to strip
       * off the xxxx part */
      if (truncate_ftp_entries)
	{
	  if (strncmp (login->ut_line, "ftp", 3) == 0)
	    {
	      login->ut_line[3] = '\0';
	    }
	}

#ifdef SYSTEM_V_STYLE_RECORDS
      /* for some reason, records get stored that seem to be of no use
       * whatsoever -- they just take up some time on a particular
       * terminal and we have no idea what they are except user
       * processes */

      if ((login->RECORD_TYPE == USER_PROCESS) && (login->ut_name[0] == '\0'))
	return;
      
      switch (login->RECORD_TYPE)
	{

	  /* in order for this program to work all right in the first
	     place, we NEED to have OLD_TIME, NEW_TIME, BOOT_TIME, and
	     USER_PROCESS, but we're not guaranteed to have anything
	     else.  Add #ifdef's accordingly. */

	case BOOT_TIME:
	  fputs ("reboot  ", stdout);
	  type = SPECIAL;
	  break;

	case OLD_TIME:
	case NEW_TIME:
	  fputs ("        ", stdout);
	  type = SPECIAL;
	  break;

#ifdef EMPTY
	case EMPTY:
	  fputs ("empty   ", stdout);
	  type = SPECIAL;
	  break;
#endif

#ifdef RUN_LVL
	case RUN_LVL:
	  if (login->ut_name[0] != '\0')
	    printf ("%-8.8s", login->ut_name);
	  else
	    fputs ("runlevel", stdout);
	  type = SPECIAL;
	  break;
#endif

#ifdef ACCOUNTING
	case ACCOUNTING:
	  if (login->ut_name[0] != '\0')
	    printf ("%-8.8s", login->ut_name);
	  else
	    fputs ("acct    ", stdout);
	  type = SPECIAL;
	  break;
#endif

#ifdef LOGIN_PROCESS
	case LOGIN_PROCESS:
	  if (login->ut_name[0] != '\0')
	    printf ("%-8.8s", login->ut_name);
	  else
	    fputs ("(login) ", stdout);
	  type = SPECIAL;
	  break;
#endif

#ifdef INIT_PROCESS
	case INIT_PROCESS:
	  if (login->ut_name[0] != '\0')
	    printf ("%-8.8s", login->ut_name);
	  else
	    fputs ("(init)  ", stdout);
	  type = SPECIAL;
	  break;
#endif

	default:
	  printf ("%-8.8s", login->ut_name);
	  break;
	}

      printf (" %-12.12s ", login->ut_line);

#else /* if ! SYSTEM_V_STYLE_RECORDS */

      printf ("%-8.8s  %-8.8s ", login->ut_name, login->ut_line);

#endif /* SYSTEM_V_STYLE_RECORDS */

      printf ("%-16.16s %-16.16s ",
#ifdef HAVE_UT_HOST
	      login->ut_host,
#else
	      "",
#endif
	      ctime ((time_t *) &login->ut_time));


      if (type == SPECIAL)
	putchar ('\n');
      else if (type == STILL_ON)
	fputs ("  still logged in\n", stdout);
      else
	{
	  char temp_str[15];
	  int days,hours,minutes;
	  int diff;
	  
	  diff = (int) logout->time - login->ut_time;
	  days = (int) (diff / 86400);
	  diff -= 86400 * days;
	  hours = (int) (diff / 3600);
	  diff -= 3600 * hours;
	  minutes = (int) (diff / 60);
	  
	  if (days)
	    sprintf (temp_str, "(%d+%02d:%02d)", days, hours, minutes);
	  else
	    sprintf (temp_str, "(%02d:%02d)", hours, minutes);
	  
	  switch (type)
	    {
	    case SHUTDOWN:
	      printf ("- down  %8s\n", temp_str);
	      break;
	    case REBOOT:
#ifdef SYSTEM_V_STYLE_RECORDS
	      printf ("- down  %8s\n", temp_str);
#else
	      printf ("- crash %8s\n", temp_str);
#endif
	      break;
	    case SUPPLANT:
	      printf ("- suppl %8s\n", temp_str);
	      break;
	    case REGULAR:
	      {
		struct tm *tmptr;
		time_t temp_time = logout->time;
		tmptr = localtime (&temp_time);
		printf ("- %02d:%02d %8s\n", tmptr->tm_hour,
			tmptr->tm_min, temp_str);
	      }
	      break;
	    default:
	      break;
	    }
	}
      if (num_lines_to_print > -1)
	{
	  num_lines_to_print--;
	  if (num_lines_to_print == 0)    /* max lines printed */
	    exit (0);
	}
    }
}



/* Do a buffered read of the file and return the next record in REC.
 * Return 0 if no more entries.
 */
int
  get_entry (struct utmp **rec)
{
  /* if BUF_LAST == 0, we know that there's nothing in the buffer,
   * so we should read another block of records from the file.  If we can't
   * read BUFFERED_RECS, read as many as possible and set up BUF_LAST
   * correctly.
   */
  
  if (buf_last == 0)
    {
      long new_offset, this_offset, max_recs, recs_read;
      
      this_offset = ftell (Wfile);
      if (this_offset == 0)
	return 0;                    /* no more records */
      
      max_recs = this_offset / (long) sizeof (struct utmp);
  
      recs_read = (max_recs < BUFFERED_RECS) ? max_recs : BUFFERED_RECS;
    
      fseek (Wfile, -sizeof (struct utmp) * recs_read, SEEK_CUR);
      new_offset = ftell (Wfile);
      
      if (debugging_enabled)
	fprintf (stddebug, "Did seek in file %ld --> %ld (%d)\n",
		 this_offset, new_offset, sizeof (struct utmp));
      
      if (fread ((void *) rec_buffer, sizeof (struct utmp),
		 recs_read, Wfile) == -1)
	fatal ("get_entry: couldn't read from file");
      
      if (debugging_enabled)
	fprintf (stddebug, "Got %ld records from file\n", recs_read);
  
      /* don't need to check this, because the above read was fine */
      fseek (Wfile, -sizeof (struct utmp) * recs_read, SEEK_CUR);
      
      buf_last = recs_read - 1;
    }		 
  else
    {
      buf_last--;
    }
  
  *rec = rec_buffer + buf_last;
  
  return 1;
}


/* clear all global flags & data */
void
init_flags_and_data ()
{
  debugging_enabled = 0;
  time_warp_leniency = 60;
  last_event = STILL_ON;
  num_lines_to_print = -1;
  print_supplant = 0;

#ifdef HAVE_LIMITS_H
  last_time = LONG_MAX;
#else
  last_time = -1;
#endif

  buf_last = 0;
  truncate_ftp_entries = 1;

  rec_buffer = (struct utmp *) xmalloc (sizeof (struct utmp) * BUFFERED_RECS);

  hash_table = NULL;
  name_list = NULL;

#ifdef SYSTEM_V_STYLE_RECORDS
  print_all_records = 0;
#endif
}

RETSIGTYPE
#if !defined (SVR4) && !defined (_POSIX_SOURCE)
handler (int sig, int code, struct sigcontext *scp)
#else
handler (int sig)
#endif
{
  printf ("\ninterrupted at %s", ctime ((time_t *) &last_time));
  exit (0);
}

