/* -*- mode: c; c-file-style: "gnu" -*-
 * daemon.c -- daemon-mode support functions
 * Copyright (C) 2002, 2003, 2004 Gergely Nagy <algernon@bonehunter.rulez.org>
 *
 * This file is part of Thy.
 *
 * Thy is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 dated June, 1991.
 *
 * Thy is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/** @file daemon.c
 * Daemon-mode support functions.
 *
 * In reality, this module contains wrappers for bind() and listen(),
 * accept() and select(). Routines to bind to and listen on many
 * addresses, to manage the I/O queue based on socket state, and to
 * occassionally wipe dead connections.
 */

#include "system.h"

#include <sys/types.h>
#include <sys/time.h>
#include <netdb.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#ifdef HAVE_STRINGS_H
#include <strings.h>
#endif
#ifdef HAVE_SYS_POLL_H
#include <sys/poll.h>
#endif
#include <sys/socket.h>
#include <unistd.h>

#include "compat/compat.h"
#include "bh-libs/list.h"

#include "cgi.h"
#include "config.h"
#include "daemon.h"
#include "misc.h"
#include "network.h"
#include "nqueue.h"
#include "options.h"
#include "queue.h"
#include "session.h"
#include "stats.h"
#include "thy.h"
#include "tls.h"
#include "types.h"

/** @internal Bind to an address.
 * This one binds to an address, handles errors and anything that may
 * arise.
 *
 * @param ai is the address information.
 * @param listen_port is the port to bind to, and listen on.
 * @param ssl signals TLS mode.
 *
 * @returns The bound filedescriptor, or -1 on error.
 */
static int
_daemon_setup_bind (const struct addrinfo *ai, in_port_t listen_port,
		    int ssl)
{
  int ret;
  const int one = 1;
  int listenfd;
  char sock_name[NI_MAXHOST];
  int port = listen_port;

#if !THY_OPTION_TLS
  if (ssl)
    {
      bhc_error ("%s", "TLS support not compiled in - "
		 "cannot accept HTTPS requests");
      return -1;
    }
#endif

  listenfd = socket (ai->ai_family, SOCK_STREAM, IPPROTO_IP);
  if (listenfd == -1)
    {
      if (errno != EAFNOSUPPORT)
	{
	  bhc_error ("socket: %s", strerror (errno));
	}
      return -1;
    }
  if (fcntl (listenfd, F_SETFD, FD_CLOEXEC) == -1)
    {
      bhc_error ("fcntl: %s", strerror (errno));
      close (listenfd);
      return -1;
    }
  if (fcntl (listenfd, F_SETFL, O_NONBLOCK) < 0)
    {
      bhc_error ("fcntl(): %s", strerror (errno));
      close (listenfd);
      return -1;
    }

  memset (sock_name, 0, sizeof (sock_name));
  getnameinfo ((struct sockaddr *)ai->ai_addr, ai->ai_addrlen, sock_name,
	       sizeof (sock_name), NULL, 0, NI_NUMERICHOST);

  switch (ai->ai_family)
    {
    case AF_INET6:
      if (port > 0)
	((struct sockaddr_in6 *)(ai->ai_addr))->sin6_port = port;
      else
	port = ((struct sockaddr_in6 *)(ai->ai_addr))->sin6_port;
      break;
    case AF_INET:
      if (port > 0)
	((struct sockaddr_in *)(ai->ai_addr))->sin_port = port;
      else
	port = ((struct sockaddr_in *)(ai->ai_addr))->sin_port;
      break;
    }

  ret = setsockopt (listenfd, SOL_SOCKET, SO_REUSEADDR,
		    &one, sizeof (one));
  if (ret != 0)
    {
      bhc_error ("setsockopt: %s (%s)", strerror (errno), sock_name);
      close (listenfd);
      return -1;
    }

  ret = bind (listenfd, ai->ai_addr, ai->ai_addrlen);
  if (ret != 0)
    {
      int np = ntohs (port);
      bhc_error ("bind to %s:%d failed: %s", sock_name, np,
		 strerror (errno));
      close (listenfd);
      return -1;
    }

  if (listen (listenfd, _THY_MAXCONN) != 0)
    {
      int np = ntohs (port);
      bhc_error ("listen on %s:%d failed: %s", sock_name, np,
		 strerror (errno));
      close (listenfd);
      return -1;
    }

  bhc_log ("Socket bound to port %d on %s%s.",
	   ntohs (port), sock_name, (ssl) ? " (ssl)" : "");

  return listenfd;
}

/** @internal Set up a listening socket.
 * Sets up a listening socket on all the required addresses. Or on all
 * possible ones, if LISTEN_ADDR is NULL.
 *
 * @param listener holds the initial listener structure.
 * @param family is the socket family: PF_INET, PF_INET6, or
 * PF_UNSPEC.
 *
 * @returns The bound FDs in a thy_listener_t array on success, NULL
 * otherwise.
 */
static thy_listener_t *
_daemon_setup_listen (thy_listener_t *listener, int family)
{
  int ret, size = 1, port, pos = 0;
  char *listen_port_str;
  struct addrinfo hints, *res, *cur;
  struct sockaddr_storage *listen_addr = NULL;
  thy_listener_t *nlisteners;
  thy_listener_ssl_t ssl;

  if (listener)
    {
      listen_addr = listener->sock;
      if (listener->ssl != LISTENER_SSL_AUTO)
	ssl = listener->ssl;
      else
	ssl = thy_default_tls;
    }
  else
    ssl = thy_default_tls;

  /* We have a specific address to listen on... */
  if (listen_addr != NULL)
    {
      cur = bhc_calloc (1, sizeof (struct addrinfo));
      cur->ai_family = listen_addr->ss_family;

      switch (cur->ai_family)
	{
	case AF_INET6:
	  cur->ai_addrlen = sizeof (struct sockaddr_in6);
	  break;
	case AF_INET:
	  cur->ai_addrlen = sizeof (struct sockaddr_in);
	  break;
	}

      cur->ai_addr = bhc_malloc (cur->ai_addrlen);
      memcpy (cur->ai_addr, listen_addr, cur->ai_addrlen);

      ret = _daemon_setup_bind (cur, 0, (ssl == LISTENER_SSL_SET));
      free (cur->ai_addr);
      free (cur);

      if (ret < 0)
	return NULL;

      nlisteners = (thy_listener_t *)
	bhc_calloc (2, sizeof (thy_listener_t));
      nlisteners[0].fd = ret;
      nlisteners[0].ssl = listener->ssl;
      return nlisteners;
    }

  /* We should listen on wherever possible */
  memset (&hints, 0, sizeof (hints));
  hints.ai_family = family;
  hints.ai_flags = AI_PASSIVE;
  hints.ai_socktype = SOCK_STREAM;

  if (thy_default_port)
    port = thy_default_port;
  else
    {
      if (ssl == LISTENER_SSL_SET)
	port = _THY_PORT_SSL;
      else
	port = _THY_PORT;
    }

  asprintf (&listen_port_str, "%d", htons (port));
  ret = getaddrinfo (NULL, listen_port_str, &hints, &res);
  free (listen_port_str);
  if (ret != 0)
    {
      bhc_error ("getaddrinfo: %s", gai_strerror (ret));
      return NULL;
    }

  if (!res)
    return NULL;

  /* Count the number of FDs we want to listen on */
  cur = res;
  do
    {
      size++;
      cur = cur->ai_next;
    } while (cur != NULL);

  /* Listen on all the addresses getaddrinfo() returned. */
  cur = res;
  nlisteners = (thy_listener_t *)
    bhc_calloc (size, sizeof (thy_listener_t));
  do
    {
      ret = _daemon_setup_bind (cur, htons (port), ssl);
      if (ret != -1)
	{
	  nlisteners[pos].fd = ret;
	  nlisteners[pos].ssl = ssl;
	  pos++;
	}
      cur = cur->ai_next;
    } while (cur != NULL);
  nlisteners[pos].fd = 0;
  freeaddrinfo (res);

  return nlisteners;
}

/** Initialise the daemon.
 * Sets up the listening sockets, and drops privileges afterwards.
 *
 * @returns A -1 terminated array of FDs Thy listens on.
 */
thy_listener_t *
daemon_init (void)
{
  const config_t *config = config_get ();
  size_t i;
  int initialised = 0;
  thy_listener_t *ls, *listeners;
  bhl_list_t *bound_fds = bhl_list_init (1, NULL);

  /* Set up the listening sockets. */
  if (bhl_list_size (config->ips) > 0)
    {
      for (i = 0; i < bhl_list_size (config->ips); i++)
	{
	  thy_listener_t *l;
	  size_t j = 0;

	  bhl_list_get (config->ips, i, (void *)&l);
	  ls = _daemon_setup_listen (l, config->type);
	  free (l);

	  if (!ls)
	    continue;

	  while (ls[j].fd)
	    {
	      bhl_list_append (bound_fds, &ls[j], sizeof (thy_listener_t));
	      j++;
	      initialised = 1;
	    }
	  free (ls);
	}
    }
  else
    {
      size_t j = 0;
      ls = _daemon_setup_listen (NULL, config->type);
      if (ls)
	{
	  while (ls[j].fd)
	    {
	      bhl_list_append (bound_fds, &ls[j], sizeof (thy_listener_t));
	      j++;
	      initialised = 1;
	    }
	  free (ls);
	}
    }

  if (config->options.chroot == THY_BOOL_TRUE)
    {
      if (setuid (config->uid))
	{
	  bhc_error ("setuid(): %s", strerror (errno));
	  bhc_exit (-1);
	}
    }
  else
    thy_priv_drop (config->uid);

  /* Convert the list to an integer array. */
  listeners = (thy_listener_t *)bhc_calloc (bhl_list_size (bound_fds) + 3,
					    sizeof (thy_listener_t));
  for (i = 0; i < bhl_list_size (bound_fds); i++)
    {
      thy_listener_t *l;

      bhl_list_get (bound_fds, i, (void **)&l);
      listeners[i].fd = l->fd;
      listeners[i].ssl = l->ssl;
      free (l->sock);
      free (l);
    }
  listeners[i].fd = -1;
  bhl_list_free (bound_fds);

  if (!initialised)
    {
      free (listeners);
      return NULL;
    }

  bhc_log ("%s", "Accepting connections...");
  return listeners;
}

/** Do a select() on our queue.
 * Sets up the file descriptor sets, and does a select() on the
 * request queue.
 *
 * In case the available fd set is running out, the listener sockets
 * are not put into the input fd_set, so new connections will not be
 * accepted until resources free up.
 *
 * @param listeners is an array of listening sockets.
 * @param timeout is the maximum number of seconds to wait for I/O.
 *
 * @returns Zero on success without pending requests, one when there
 * is a pending request, -1 on error.
 *
 * @note As a side-effect, it modifies the network I/O queue.
 */
int
daemon_select (thy_listener_t *listeners, long timeout)
{
  int lfdn = 0;
  long int i;

  for (i = 0; listeners[i].fd > 0; i++)
    {
      thy_nq_fd_control (listeners[i].fd, THY_NQ_EVENT_NONE, 0);
      lfdn++;
    }

  if (thy_nq_has_pending ())
    return 1;

  /* Add listeners to the input queue, but only if we have enough
     FDs. */
  if (thy_nq_max () < _THY_MAXCONN - lfdn - _THY_MAXACCEPT - 4)
    for (i = 0; listeners[i].fd > 0; i++)
      thy_nq_fd_control (listeners[i].fd, THY_NQ_EVENT_INPUT, 1);
  else
    {
      bhc_debug ("%s",
		 "FD shortage. Listeners omitted from the input queue.");
    }

  /* Do the select magic */
  if (thy_nq_act () > 0)
    {
      if (thy_nq_wait (timeout) == -1)
	{
	  if (errno != EINTR && errno != 0 && errno != EBADF)
	    {
	      bhc_error ("thy_nq_wait(): %s", strerror (errno));
	      return -1;
	    }
	}
    }

  return 0;
}

/** Accept new connections, and place them in the queue.
 * This function accepts as many connections (from all listeners, in a
 * round-robin fashion) as it can, but not more than #_THY_MAXCONN.
 *
 * Accepting connections in batches is a good thing, since it improves
 * performance a great deal on heavy load, since ready connections
 * don't have to sit there for a daemon_handle_() + daemon_select()
 * iteration.
 *
 * @param listeners is an array of listening FDs.
 *
 * @returns Zero on success, -1 on failure.
 *
 * @note As a side-effect, it adds a new entry to the queue if the
 * connection was accepted.
 */
int
daemon_accept (thy_listener_t *listeners)
{
  int rs, i, accept_loop, accepted = 0;
  struct sockaddr_storage faddr;
  socklen_t socklen = sizeof (struct sockaddr_storage);
  char peer_name[NI_MAXHOST];
#if HAVE_DECL_TCP_CORK
  int o;
#endif

  for (i = 0; listeners[i].fd > 0; i++)
    {
      if (thy_nq_fd_check (listeners[i].fd, THY_NQ_EVENT_INPUT))
	{
	  accept_loop = 1;
	  while (accept_loop && accepted < _THY_MAXACCEPT)
	    {
	      /* Attempt to accept the connection */
	      memset (&faddr, 0, sizeof (struct sockaddr_storage));

	      if ((rs = accept (listeners[i].fd, (struct sockaddr *)&faddr,
				&socklen)) < 0)
		{
		  if (errno == EWOULDBLOCK)
		    {
		      accept_loop = 0;
		      break;
		    }
		  bhc_error ("accept(): %s", strerror (errno));
		  return -1;
		}
	      accepted++;

	      /* Do we have enough free slots? */
	      if (rs > _THY_MAXCONN || rs > (int)FD_SETSIZE)
		{
		  bhc_error ("%s", "Too many open connections.");
		  close (rs);
		  return -1;
		}

	      /* Set non-blocking mode */
	      if (fcntl (rs, F_SETFL, O_NONBLOCK) < 0)
		{
		  bhc_error ("fcntl(): %s", strerror (errno));
		  close (rs);
		  return -1;
		}

	      /* Close-on-exec */
	      if (fcntl (rs, F_SETFD, FD_CLOEXEC) == -1)
		{
		  bhc_error ("fcntl: %s", strerror (errno));
		  close (rs);
		  return -1;
		}

#if HAVE_DECL_TCP_CORK
	      /* TCP_CORK */
	      o = 1;
	      setsockopt (rs, SOL_SOCKET, TCP_CORK, &o, sizeof (o));
#endif

	      if (getnameinfo ((struct sockaddr *)&faddr, socklen,
			       peer_name, sizeof (peer_name), NULL,
			       0, NI_NUMERICHOST))
		strcpy (peer_name, "0.0.0.0");

	      bhc_debug ("Accepted connection from %s on %d",
			 peer_name, rs);

	      /* Add it to the input queue */
	      queue_add (session_init
			 (rs, rs, thy_sock_port_get (listeners[i].fd),
			  (listeners[i].ssl == LISTENER_SSL_SET),
			  peer_name, NULL));
	    }
	  bhc_debug ("Accepted connections on %d: %d", listeners[i].fd,
		     accepted);
	  thy_stats_requests_inc (accepted);
	}
    }
  return 0;
}

/** @internal Wrapper around read()/thy_recv().
 * Based on the type of the session (CGI or not), this function passes
 * control on to either read() or thy_recv().
 */
static int
_daemon_recv (session_t *session, void *buffer, size_t length)
{
  if (session->cgi.pipes.in[0] > 0)
    return read (session->cgi.pipes.in[0], buffer, length);
  else
    return thy_recv (session, buffer, length);
}

/** Handle all socket I/O.
 * This is a bit complex, since it handles all socket I/O, including
 * input from a socket.
 *
 * @note As a side-effect, it will modify the I/O queue.
 */
void
daemon_handle_io (void)
{
  int i, r;
  session_t *session;
  char c;
  long int nq_max = thy_nq_max ();
  const config_t *config = config_get ();

  thy_nq_reset ();

  for (i = 0; i <= nq_max; i++)
    {
      session = queue_get (i);
      if (!session)
	continue;

#if THY_OPTION_TLS
      /* Handshake */
      if (thy_nq_fd_check (session->io.in, THY_NQ_EVENT_INPUT) &&
	  session->state == SESSION_STATE_HANDSHAKE)
	{
	  if (!thy_tls_session_init (session))
	    queue_del (i);
	  continue;
	}
#endif

      /* Process a pending session. */
      if (session->state == SESSION_STATE_PROCESS)
	{
	  if (!session->request)
	    session_update (session);
	  if (session_handle (session))
	    queue_del (i);
	  else
	    {
	      if (session->state == SESSION_STATE_PROCESS)
		session_state_change (session, SESSION_STATE_OUTPUT_HEAD);
	      session_finalise (session);
	    }
	  continue;
	}

      /* Push output */
      if (thy_nq_fd_check (session->io.out, THY_NQ_EVENT_OUTPUT) &&
	  (session->state == SESSION_STATE_OUTPUT_HEAD ||
	   session->state == SESSION_STATE_OUTPUT_BODY ||
	   session->state == SESSION_STATE_CGI_HEADER_OUTPUT ||
	   session->state == SESSION_STATE_CGI_BODY_OUTPUT ||
	   session->state == SESSION_STATE_OUTPUT_100 ||
	   session->state == SESSION_STATE_OUTPUT_101))
	{
	  if (session->state == SESSION_STATE_CGI_BODY_OUTPUT ||
	      session->state == SESSION_STATE_OUTPUT_BODY)
	    {
	      if (session->chunked.enabled &&
		  session->chunked.prechunk.size == 0)
		{
		  free (session->chunked.prechunk.buff);
		  session->chunked.prechunk.offset = 0;
		  if (session->body.content_size > 0)
		    asprintf (&session->chunked.prechunk.buff, "%x\r\n",
			      (unsigned int)session->body.content_size);
		  else
		    session->chunked.prechunk.buff = bhc_strdup ("0\r\n");
		  session->chunked.prechunk.size =
		    strlen (session->chunked.prechunk.buff);

		  free (session->chunked.postchunk.buff);
		  session->chunked.postchunk.buff = bhc_strdup ("\r\n");
		  session->chunked.postchunk.offset = 0;
		  session->chunked.postchunk.size = 2;
		}
	    }
	  if ((*session->throttle) (session))
	    {
	      if (session->state == SESSION_STATE_CGI_HEADER_OUTPUT &&
		  session->request->method != HTTP_METHOD_HEAD &&
		  session->request->method != HTTP_METHOD_OPTIONS)
		{
		  session_state_change (session,
					SESSION_STATE_CGI_BODY_INPUT);
		  free (session->body.buffer);
		  session->body.buffer = NULL;
		}
	      else if (session->state == SESSION_STATE_CGI_BODY_OUTPUT)
		{
		  session->chunked.prechunk.size = 0;
		  if (session->cgi.eof)
		    {
		      if (session_handle_keepalive (session))
			queue_del (i);
		    }
		  else
		    {
		      free (session->body.buffer);
		      session->body.buffer = NULL;
		      session->body.offset = 0;
		      session_state_change (session,
					    SESSION_STATE_CGI_BODY_INPUT);
		    }
		}
	      else if (session->state == SESSION_STATE_OUTPUT_100)
		{
		  free (session->request->expect);
		  session->request->expect = NULL;
		  session_state_change (session, SESSION_STATE_PROCESS);
		}
	      else if (session->state == SESSION_STATE_OUTPUT_101)
		{
#if THY_OPTION_TLS
		  switch (session->request->upgrade)
		    {
		    case THY_UPGRADE_TLS10:
		      session->ssl = 1;
		      session_state_change (session,
					    SESSION_STATE_HANDSHAKE);
		      session->tls_session = NULL;
		      if (!thy_tls_session_init (session))
			{
			  queue_del (i);
			  continue;
			}
		      session->request->upgrade = THY_UPGRADE_TLS10_POST;
		      continue;
		    default:
		      break;
		    }
#endif

		  session->request->upgrade = THY_UPGRADE_NONE;
		  session_state_change (session, SESSION_STATE_PROCESS);
		}
	      else if (session_handle_keepalive (session))
		  queue_del (i);
	    }
	  else
	    session->acttime = time (NULL); /* Zero out timeout counter */

	  continue;
	}

      /* Accept input (all but CGI/POST body) */
      if ((session->state == SESSION_STATE_INPUT_REQUEST &&
	   thy_nq_fd_check (session->io.in, THY_NQ_EVENT_INPUT)) ||
	  (session->state == SESSION_STATE_CGI_HEADER_INPUT &&
	   thy_nq_fd_check (session->cgi.pipes.in[0], THY_NQ_EVENT_INPUT)))
	{
	  int error = 0;

	  /* Zero out timeout counter */
	  session->acttime = time (NULL);

	  /* Read HTTP request */
	  if (!session->body.buffer)
	    {
	      session->body.len = 0;
	      session->body.size = 1024;
	      session->body.buffer =
		(char *)bhc_malloc (session->body.size);
	    }

	  while ((r = _daemon_recv (session, &c, 1)) >= 0)
	    {
	      if (session->body.len >= config->limits.header &&
		  session->state == SESSION_STATE_INPUT_REQUEST &&
		  config->limits.header != 0)
		{
		  bhc_log ("Client %s sent a request too large on %d.",
			   session->origin, session->io.in);

		  session->request =
		    (request_t *)bhc_calloc (1, sizeof (request_t));
		  session->request->http_major = 1;
		  session->request->http_minor = 0;

		  free (session->body.buffer);
		  session->body.buffer = NULL;

		  session_header (session, HTTP_STATUS_413);
		  session_error (session);
		  session_state_change (session,
					SESSION_STATE_OUTPUT_HEAD);
		  session_finalise (session);
		  error = 1;

		  break;
		}

	      if (session->body.len + 2 >= session->body.size)
		{
		  session->body.size *= 2;
		  session->body.buffer = (char *)bhc_realloc
		    (session->body.buffer, session->body.size);
		}
	      if (r > 0)
		{
		  session->body.buffer[session->body.len++] = c;
		  session->body.buffer[session->body.len] = 0;
		}

	      if (session->body.len > 2 &&
		  session->body.buffer[session->body.len-2] == '\n' &&
		  session->body.buffer[session->body.len-1] == '\n')
		break;
	      if (session->body.len > 4 &&
		  session->body.buffer[session->body.len-4] == '\r' &&
		  session->body.buffer[session->body.len-3] == '\n' &&
		  session->body.buffer[session->body.len-2] == '\r' &&
		  session->body.buffer[session->body.len-1] == '\n')
		break;

	      if (r == 0)
		{
		  if (session->state == SESSION_STATE_INPUT_REQUEST)
		    {
		      bhc_log ("Unexpected end of headers from %s (%d)",
			       session->origin, session->io.in);
		    }
		  else
		    {
		      bhc_log ("Premature end of headers (%d) from %s",
			       session->io.in, session->origin);
		    }
		  break;
		}
	    }

	  if (error)
	    continue;

	  if (r >= 0)
	    {
	      if (session->state == SESSION_STATE_INPUT_REQUEST)
		{
		  if (r == 0)
		    {
		      queue_del (i);
		      continue;
		    }

		  session_state_change (session, SESSION_STATE_PROCESS);
		}
	      else
		{
		  if (r == 0)
		    {
		      session_state_change (session,
					    SESSION_STATE_OUTPUT_HEAD);
		      close (session->cgi.pipes.in[0]);
		      close (session->cgi.pipes.out[1]);
		      session->cgi.pipes.in[0] = -1;
		      free (session->body.buffer);
		      session->body.buffer = NULL;

		      kill (session->cgi.child, SIGKILL);
		      session->cgi.running = -1;
		      session->cgi.child = -1;
		      free (session->cgi.handler);
		      session->cgi.handler = NULL;

		      session_header (session, HTTP_STATUS_500);
		      session_error (session);
		      session_finalise (session);
		      continue;
		    }
		  session_state_change (session,
					SESSION_STATE_CGI_HEADER_OUTPUT);
		  cgi_headers_setup (session);
		  session->body.content_size = session->body.len;
		  session->body.content_length = session->body.len;
		  session->body.offset = 0;
		  session_finalise (session);
		}
	    }
	  else
	    {
	      const char *err = NULL;
#if THY_OPTION_TLS
	      if (session->state == SESSION_STATE_INPUT_REQUEST &&
		  thy_tls_istls (session) && gnutls_error_is_fatal (r))
		err = gnutls_strerror (r);
#endif
	      if (errno != EAGAIN && errno != EWOULDBLOCK)
		err = strerror (errno);

	      if (err)
		{
		  bhc_log ("Unexpected error from %s (%d): %s",
			   session->origin, i, err);
		  queue_del (i);
		}
	    }
	  continue;
	}

      /* Accept CGI body */
      if (session->state == SESSION_STATE_CGI_BODY_INPUT &&
	  thy_nq_fd_check (session->cgi.pipes.in[0], THY_NQ_EVENT_INPUT))
	{
	  /* Zero out timeout counter */
	  session->acttime = time (NULL);

	  session->body.len = 0;
	  session->body.size = _THY_BUFSIZE;
	  session->body.buffer =
	    (char *)bhc_malloc (session->body.size + 1);

	  session->cgi.eof = 1;
	  r = _daemon_recv (session, session->body.buffer,
			    session->body.size);

	  if (r >= 0)
	    {
	      session_state_change (session,
				    SESSION_STATE_CGI_BODY_OUTPUT);

	      if (r > 0)
		session->cgi.eof = 0;

	      session->body.len = r;
	      session->body.content_size = r;
	      session->body.content_length = r;
	      session->body.offset = 0;
	      session->cgi.content_length += r;
	      session_finalise (session);
	    }
	  else
	    {
	      if (errno != EAGAIN && errno != EWOULDBLOCK)
		{
		  bhc_log ("Unexpected error from %s (%d): %s",
			   session->origin, i, strerror (errno));
		  queue_del (i);
		}
	    }
	  continue;
	}

      /* Accept POST body */
      if ((session->state == SESSION_STATE_POST_INPUT ||
	   session->state == SESSION_STATE_INPUT_DISCARD) &&
	  thy_nq_fd_check (session->io.in, THY_NQ_EVENT_INPUT))
	{
	  size_t rl = _THY_BUFSIZE;

	  /* Zero out timeout counter */
	  session->acttime = time (NULL);

	  free (session->cgi.post.buffer);
	  if (session->request->content_length < _THY_BUFSIZE)
	    rl = session->request->content_length;
	  session->cgi.post.size = rl;
	  session->cgi.post.buffer =
	    (char *)bhc_malloc (rl);

	  r = thy_recv (session, session->cgi.post.buffer, rl);

	  if (r == 0)
	    {
	      bhc_log ("Unexpected end of POST data on %d from %s",
		       session->io.in, session->origin);
	      queue_del (i);
	      continue;
	    }

	  if (r > 0)
	    {
	      session->cgi.post.length += r;
	      session->cgi.post.len = r;

	      if (session->state == SESSION_STATE_INPUT_DISCARD)
		{
		  if (session->cgi.post.length >=
		      session->request->content_length)
		    session_state_change (session,
					  SESSION_STATE_PROCESS);
		  continue;
		}

	      session_state_change (session, SESSION_STATE_POST_OUTPUT);

	      session->cgi.post.offset = 0;
	      session_finalise (session);
	    }
	  else
	    {
	      if (errno != EAGAIN && errno != EWOULDBLOCK)
		{
		  bhc_log ("Unexpected error from %s (%d): %s",
			   session->origin, i, strerror (errno));
		  queue_del (i);
		}
	    }
	  continue;
	}

      /* Push POST body */
      if (session->state == SESSION_STATE_POST_OUTPUT &&
	  thy_nq_fd_check (session->cgi.pipes.out[1], THY_NQ_EVENT_OUTPUT))
	{
	  int v = -1;

	  r = -1;

	  /* Zero out timeout counter */
	  session->acttime = time (NULL);

	  if ((size_t) session->cgi.post.offset < session->cgi.post.len &&
	      session->cgi.post.len > 0)
	    r = write (session->cgi.pipes.out[1],
		       &session->cgi.post.buffer[session->cgi.post.offset],
		       session->cgi.post.len - session->cgi.post.offset);

	  if (r >= 0)
	    {
	      session->cgi.post.offset += r;
	      if ((size_t)session->cgi.post.offset >=
		  session->cgi.post.size)
		v = 0;
	    }

	  if (v == -1)
	    {
	      if (session->cgi.post.length >=
		  session->request->content_length)
		{
		  session_state_change (session,
					SESSION_STATE_CGI_HEADER_INPUT);
		  close (session->cgi.pipes.out[1]);
		}
	      else
		session_state_change (session, SESSION_STATE_POST_INPUT);
	    }

	  continue;
	}

      /* Handle non-live connections */

      /* Did it time out? */
      if (time (NULL) - session->acttime > config->timeout ||
	  (session->request_count > 0 &&
	   time (NULL) - session->acttime > config->ktimeout))
	{
	  bhc_log ("Connection timed out from %s (%d)", session->origin,
		   session->io.in);
	  queue_del (i);
	  continue;
	}

      /* Is the remote end still there at all? */
      if (thy_socket_is_alive (session->io.out) != 0)
	{
	  bhc_log ("Connection from %s (%d) closed by remote end",
		   session->origin, session->io.in);
	  queue_del (i);
	  continue;
	}

      session_state_change (session, session->state);
    }
}
