#include "CORBAmodule.h"
#include <signal.h>

/***************************************************************************
	Wraps CORBA_ORB
	Methods handled:
		object_to_string
		string_to_object
		run
*/

POA_PyObject *root_poa;
GHashTable *orb_objects;

#ifndef ORBIT_PYTHON_NOT_THREADED
PyThreadState *python_thread_state;
#endif /* ORBIT_PYTHON_NOT_THREADED */

PyTypeObject CORBA_ORB_PyObject_Type = {
	PyObject_HEAD_INIT(NULL)
	0,       /*ob_size*/
	"CORBA.ORB",         /*tp_name*/
	sizeof(CORBA_ORB_PyObject),   /*tp_basicsize*/
	0,       /*tp_itemsize*/
	/* methods */
	(destructor)CORBA_ORB_PyObject__dealloc, /*tp_dealloc*/
	0,       /*tp_print*/
	(getattrfunc)CORBA_ORB_PyObject__getattr, /*tp_getattr*/
	(setattrfunc)CORBA_ORB_PyObject__setattr, /*tp_setattr*/
	0,       /*tp_compare*/
	0,       /*tp_repr*/
	0,       /*tp_as_number*/
	0,       /*tp_as_sequence*/
	0,       /*tp_as_mapping*/
	0,       /*tp_hash*/
};

CORBA_ORB_PyObject *CORBA_ORB_PyObject__new(CORBA_ORB orb)
{
	CORBA_ORB_PyObject *self;

	self = g_hash_table_lookup(orb_objects, orb);
	if (self) {
		Py_INCREF(self);
		return self;
	}

	self = PyObject_NEW(CORBA_ORB_PyObject, &CORBA_ORB_PyObject_Type);
	if (!self)
		return NULL;

	CORBA_exception_init(&self->ev);

	self->obj = orb;
	g_hash_table_insert(orb_objects, orb, self);
	return self;
}

void CORBA_ORB_PyObject__dealloc(CORBA_ORB_PyObject *self)
{
	CORBA_Object_release((CORBA_Object)self->obj, &self->ev);
	CORBA_exception_free(&self->ev);
	g_hash_table_remove(orb_objects, self->obj);
	PyMem_DEL(self);
}

PyObject *CORBA_ORB_PyObject__object_to_string(CORBA_ORB_PyObject *self, 
                                               PyObject *args)
{
	PyObject *obj, *strobj = NULL;
	CORBA_PyInstance_Glue *glue;
	CORBA_Object corba_obj;
	CORBA_char *s;

	if (!PyArg_ParseTuple(args, "O:object_to_string", &obj))
		return NULL;

	// check to see if it's a known objref
	glue = g_hash_table_lookup(object_instance_glue, obj);
	if (!glue) {
		raise_system_exception(OPExc_BAD_PARAM, 0, CORBA_COMPLETED_NO, NULL);
		return NULL;
	}
	corba_obj = glue->obj;


	s = CORBA_ORB_object_to_string(self->obj, corba_obj, &self->ev);
	if (check_corba_ex(&self->ev))
		strobj = PyString_FromString(s);
	if (s)
 		CORBA_free(s);

	return strobj;
}

PyObject *CORBA_ORB_PyObject__string_to_object(CORBA_ORB_PyObject *self, 
                                               PyObject *args)
{
	CORBA_Object obj;
	char *s, *p;
	if (!PyArg_ParseTuple(args, "s", &s))
		return NULL;
	// Remove traililng whitespace from IOR
	p = s + strlen(s) - 1;
	while (isspace(*p)) 
		*p-- = 0;

	obj = CORBA_ORB_string_to_object(self->obj, s, &self->ev);
	if (!check_corba_ex(&self->ev))
		return NULL;
	if (obj == CORBA_OBJECT_NIL) {
		Py_INCREF(Py_None);
		return Py_None;
	}
	return ((PyObject *)CORBA_Object_to_PyObject(obj));
}

PyObject *CORBA_ORB_PyObject__resolve_initial_references(
                                   CORBA_ORB_PyObject *self, PyObject *args)
{
	CORBA_Object res;
	char *s;
	if (!PyArg_ParseTuple(args, "s", &s))
		return NULL;

	// workaround for ORBit bug
	// http://bugzilla.gnome.org/show_bug.cgi?id=55726
	// FIXME: should raise CORBA.ORB.InvalidName

	if (strcmp(s, "RootPOA") && strcmp(s, "NameService") &&
	    strcmp(s, "ImplementationRepository") && 
	    strcmp(s, "InterfaceRepository"))
		return raise_system_exception(OPExc_BAD_PARAM, 0, CORBA_COMPLETED_NO, 
		                              "Invalid name");

	res = CORBA_ORB_resolve_initial_references(self->obj, s, &self->ev);
	if (!check_corba_ex(&self->ev)) 
		return NULL;
	if (!strcmp(s, "RootPOA")) {
		if (!root_poa)
			root_poa = POA_Object_to_PyObject((PortableServer_POA)res);
		Py_INCREF(root_poa);
		return (PyObject *)root_poa;
	}
	return ((PyObject *)CORBA_Object_to_PyObject(res));
}


PyObject *CORBA_ORB_PyObject__run(CORBA_ORB_PyObject *self, PyObject *args)
{
	void (*oldsig)(int);

	if (!PyArg_ParseTuple(args, ""))
		return NULL;

	// Make ctrl-c work while the ORB is running.
	oldsig = signal(SIGINT, SIG_DFL);

#ifndef ORBIT_PYTHON_NOT_THREADED
	python_thread_state = PyEval_SaveThread();
#endif /* ORBIT_PYTHON_NOT_THREADED */
	CORBA_ORB_run(self->obj, &self->ev);
#ifndef ORBIT_PYTHON_NOT_THREADED
	PyEval_RestoreThread(python_thread_state);
#endif /* ORBIT_PYTHON_NOT_THREADED */
	signal(SIGINT, oldsig);
	if (!check_corba_ex(&self->ev))
		return NULL;

	Py_INCREF(Py_None);
	return Py_None;
}

PyObject *CORBA_ORB_PyObject__shutdown(CORBA_ORB_PyObject *self, PyObject *args)
{
	short wait;
	if (!PyArg_ParseTuple(args, "h", &wait))
		return NULL;
	// FIXME: probably don't allow this to be called unless the ORB is really
	// running?  Seems to cause segfaults.
	CORBA_ORB_shutdown(self->obj, (CORBA_boolean)wait, &self->ev);
	if (!check_corba_ex(&self->ev))
		return NULL;

	Py_INCREF(Py_None);
	return Py_None;
}

PyMethodDef CORBA_ORB_PyObject_methods[] = {
	{ "object_to_string", (PyCFunction)CORBA_ORB_PyObject__object_to_string, 1 },
	{ "string_to_object", (PyCFunction)CORBA_ORB_PyObject__string_to_object, 1 },
	{ "resolve_initial_references", 
	  (PyCFunction)CORBA_ORB_PyObject__resolve_initial_references, 1 },
	{ "run", (PyCFunction)CORBA_ORB_PyObject__run, 1},
	{ "shutdown", (PyCFunction)CORBA_ORB_PyObject__shutdown, 1},
	{ NULL, NULL }
};

PyObject *CORBA_ORB_PyObject__getattr(CORBA_ORB_PyObject *self, char *name)
{
//	self->last = g_strdup(name);
	return Py_FindMethod(CORBA_ORB_PyObject_methods, (PyObject *)self, name);
}

int CORBA_ORB_PyObject__setattr(CORBA_ORB_PyObject *self, char *n, PyObject *v)
{
	return -1;
}

/* Local Variables: */
/* c-file-style: "python" */
/* End: */
